% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/check_input_validity.R
\encoding{UTF-8}
\name{check_input_validity}
\alias{check_input_validity}
\title{Validate Time and Group Inputs for Case Comparison / Eingabepruefung fuer Zeit- und Gruppierungsvariablen}
\usage{
check_input_validity(
  months,
  years,
  shift_month,
  granularity,
  agg_fun,
  df,
  group_col = NULL
)
}
\arguments{
\item{months}{Integer vector of months (1:12). / Vektor der Monate (1:12).}

\item{years}{Integer vector of years (must be strictly increasing). / Vektor der Jahre (streng aufsteigend).}

\item{shift_month}{One of "none", "mth_to_next", "mth_to_prev"; defines cross-year logic. / Jahreslogik.}

\item{granularity}{"day" or "week". / Aggregationsebene.}

\item{agg_fun}{Aggregation function: "sum", "mean", or "median". / Aggregationsfunktion.}

\item{df}{Data frame used to validate group_col. / Datensatz zur Validierung von group_col.}

\item{group_col}{Optional grouping column(s) to validate. / Optionale Gruppierungsvariable(n).}
}
\value{
A list with standardized values for:
\describe{
  \item{months}{Validated months vector}
  \item{years}{Validated and sorted years}
  \item{granularity}{One of "day" or "week"}
  \item{agg_fun}{One of "sum", "mean", or "median"}
  \item{shift_month}{Cross-year setting}
}
}
\description{
This function checks the validity of time-based and grouping arguments passed to functions like compare_monthly_cases().
It validates month/year ranges, aggregation settings, and optionally the presence and structure of group_col,
returning standardized values and user-friendly messages for potential issues (e.g., non-factors or too many levels).
}
\details{
Diese Funktion prueft Zeit- und Gruppierungsparameter, wie sie z.B. in compare_monthly_cases() verwendet werden.
Sie validiert Monats- und Jahresangaben, Aggregationseinstellungen und (optional) die Struktur von group_col,
und gibt standardisierte Werte sowie Hinweise bei potenziellen Problemen (z.B. fehlende Faktoren, zu viele Gruppen) zurueck.


Function Behavior and Messages:
- Issues stop() for invalid months/years or aggregation settings.
- Warns if group_col is missing in df.
- Gives messages if group variables are not factors, or if too many levels (>8) are detected.

Funktionsverhalten und Hinweise:
- Bei ungueltigen Zeitangaben erfolgt ein Abbruch (stop()).
- Warnung bei nicht vorhandenen Gruppierungsvariablen.
- Hinweis, falls Gruppenvariablen keine Faktoren sind oder zu viele Auspraegungen (>8) besitzen.
}
\examples{
# Example 1: Valid input without group_col
# Beispiel 1: Gueltige Eingabe ohne Gruppenvariable
df <- data.frame(
  datum = seq.Date(from = as.Date("2023-12-01"), to = as.Date("2025-02-28"), by = "day"),
  neue_faelle = sample(0:100, 456, replace = TRUE)
)
check_input_validity(
  months = c(12, 1, 2),
  years = c(2024, 2025),
  shift_month = "mth_to_next",
  granularity = "day",
  agg_fun = "sum",
  df = df
)

# Example 2: group_col exists but is not a factor
# Beispiel 2: group_col ist kein Faktor: Hinweis wird ausgegeben
df$region <- sample(c("Nord", "Sued", "West"), size = nrow(df), replace = TRUE)
check_input_validity(
  months = c(12, 1, 2),
  years = c(2024, 2025),
  shift_month = "mth_to_next",
  granularity = "day",
  agg_fun = "mean",
  df = df,
  group_col = "region"
)

# Example 3: Too many group levels triggers a message
# Beispiel 3: Zu viele Gruppenauspraegungen (>8): Warnung zur Plot-Lesbarkeit
df$gruppe <- factor(paste0("G", sample(1:12, size = nrow(df), replace = TRUE)))
check_input_validity(
  months = c(12, 1, 2),
  years = c(2024, 2025),
  shift_month = "mth_to_next",
  granularity = "week",
  agg_fun = "median",
  df = df,
  group_col = "gruppe"
)

}
\seealso{
[compare_monthly_cases()], [run_group_tests()]

Diese Funktion wird typischerweise zusammen mit [compare_monthly_cases()], [run_group_tests()] verwendet.
}
