% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mesh.R
\name{add_mesh_covariates}
\alias{add_mesh_covariates}
\title{Add mesh covariates to vertices and triangles}
\usage{
add_mesh_covariates(
  mesh,
  data,
  covariates,
  coords,
  power = 2,
  method = c("gstat", "rann"),
  k = 10,
  barrier = NULL
)
}
\arguments{
\item{mesh}{A mesh object from \pkg{fmesher} or \pkg{sdmTMB} (for
\code{\link[sdmTMB:sdmTMB]{sdmTMB::sdmTMB()}} models only).}

\item{data}{A data frame with coordinate columns and covariate columns, or an
\pkg{sf} object.}

\item{covariates}{Character vector of covariate column names to interpolate.}

\item{coords}{Character vector of coordinate column names. Ignored if data is
an \pkg{sf} object.}

\item{power}{Numeric power parameter for inverse distance weighting (default
\code{2}; Euclidean squared decay).}

\item{method}{Interpolation method. Options: \code{"gstat"} (default, exact
inverse distance weighting using gstat package) or \code{"rann"} (fast k-nearest
neighbours inverse distance weighting using \pkg{RANN} package for very
large datasets).}

\item{k}{Number of nearest neighbours to use for \code{"rann"} method (default
\code{10}). Ignored for \code{"gstat"} method.}

\item{barrier}{Optional \pkg{sf} polygon object defining barrier regions. If
provided, adds a logical \code{barrier} column to both \code{vertex_covariates} and
\code{triangle_covariates}. E.g., in the case of modelling fish in the ocean,
\code{TRUE} represents vertices/triangle centers over land and \code{FALSE} represents
vertices/triangle centers over water. For triangles, also adds a
\code{barrier_proportion} column indicating the proportion of each triangle's area
that intersects with the barrier polygon (0 = no intersection, 1 = triangle
completely within barrier).}
}
\value{
Modified mesh object with \code{vertex_covariates} and \code{triangle_covariates}
elements added and class \code{vertex_cov} added. The \code{vertex_covariates} data frame
contains covariate values interpolated at mesh vertices, and \code{triangle_covariates}
contains covariate values interpolated at triangle centers.
}
\description{
Interpolates covariate values from a data frame to mesh vertices using
inverse distance weighting (IDW). Uses \pkg{gstat} for exact IDW
interpolation by default, with an optional high-performance \pkg{RANN} method
for very large datasets.
}
\examples{
\dontshow{if (requireNamespace("RANN", quietly = TRUE)) withAutoprint(\{ # examplesIf}
library(sdmTMB)
library(sf)

# Regular data frame
mesh <- fmesher::fm_mesh_2d(pcod[, c("X", "Y")], cutoff = 10)
mesh_with_covs <- add_mesh_covariates(
  mesh,
  data = qcs_grid,
  covariates = c("depth"),
  coords = c("X", "Y")
)
head(mesh_with_covs$vertex_covariates)

# Visualize what we've done:
if (requireNamespace("ggplot2", quietly = TRUE)) {
  library(ggplot2)
  df <- as.data.frame(mesh_with_covs$loc[,1:2])
  df <- cbind(df, mesh_with_covs$vertex_covariates)
  ggplot() +
    geom_raster(data = qcs_grid, aes(X, Y, fill = depth), alpha = 0.7) +
    geom_point( data = df, aes(V1, V2, fill = depth),
                colour = "#00000010", pch = 21) +
    scale_fill_viridis_c(option = "G", trans = "log", direction = -1)

  df_tri <- mesh_with_covs$triangle_covariates
  ggplot() +
    geom_raster( data = qcs_grid, aes(X, Y, fill = depth), alpha = 0.7) +
    geom_point( data = df_tri, aes(x = .x_triangle, y = .y_triangle, fill = depth),
                colour = "#00000010", pch = 21) +
    scale_fill_viridis_c(option = "G", trans = "log", direction = -1)
}

# Piped version
mesh_with_covs <- fmesher::fm_mesh_2d(pcod[, c("X", "Y")], cutoff = 10) |>
  add_mesh_covariates(
    qcs_grid,
    covariates = c("depth_scaled", "depth_scaled2"),
    coords = c("X", "Y")
  )

# With sf objects (coords automatically extracted)
pcod_sf <- st_as_sf(pcod, coords = c("X", "Y"))
grid_sf <- st_as_sf(qcs_grid, coords = c("X", "Y"))
mesh_sf <- fmesher::fm_mesh_2d(pcod_sf, cutoff = 10) |>
  add_mesh_covariates(grid_sf, c("depth"))

# With sdmTMB mesh (coordinate names and mesh automatically detected)
mesh <- make_mesh(pcod, c("X", "Y"), cutoff = 10) |>
  add_mesh_covariates(qcs_grid, c("depth"))

# Use RANN method for very large datasets (much faster)
mesh_fast <- fmesher::fm_mesh_2d(pcod[, c("X", "Y")], cutoff = 10) |>
  add_mesh_covariates(
    qcs_grid,
    covariates = c("depth_scaled", "depth_scaled2"),
    coords = c("X", "Y"),
    method = "rann",
    k = 15
  )
\dontshow{\}) # examplesIf}
}
