% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vec-slidify.R
\name{slidify_vec}
\alias{slidify_vec}
\title{Rolling Window Transformation}
\usage{
slidify_vec(
  .x,
  .f,
  ...,
  .period = 1,
  .align = c("center", "left", "right"),
  .partial = FALSE
)
}
\arguments{
\item{.x}{A vector to have a rolling window transformation applied.}

\item{.f}{A summary \verb{[function / formula]}
\itemize{
\item If a \strong{function}, e.g. \code{mean}, the function is used with any
additional arguments, \code{...}.
\item If a \strong{formula}, e.g. \code{~ mean(., na.rm = TRUE)}, it is converted to a function.
}

This syntax allows you to create very compact anonymous functions.}

\item{...}{Additional arguments passed on to the \code{.f} function.}

\item{.period}{The number of periods to include in the local rolling window.
This is effectively the "window size".}

\item{.align}{One of "center", "left" or "right".}

\item{.partial}{Should the moving window be allowed to return partial (incomplete) windows instead of \code{NA} values.
Set to FALSE by default, but can be switched to TRUE to remove \code{NA}'s.}
}
\value{
A numeric vector
}
\description{
\code{slidify_vec()} applies a \emph{summary function} to a rolling sequence of windows.
}
\details{
The \code{slidify_vec()} function is a wrapper for \code{slider::slide_vec()} with parameters
simplified "center", "left", "right" alignment.

\strong{Vector Length In == Vector Length Out}

\code{NA} values or \code{.partial} values
are always returned to ensure the length of the return vector
is the same length of the incoming vector. This ensures easier use with \code{dplyr::mutate()}.

\strong{Alignment}

Rolling functions generate \code{.period - 1} fewer values than the incoming vector.
Thus, the vector needs to be aligned. Alignment of the vector follows 3 types:
\itemize{
\item \strong{Center:} \code{NA} or \code{.partial} values are divided and added to the beginning and
end of the series to "Center" the moving average.
This is common for de-noising operations. See also \verb{[smooth_vec()]} for LOESS without NA values.
\item \strong{Left:} \code{NA} or \code{.partial} values are added to the end to shift the series to the Left.
\item \strong{Right:} \code{NA} or \code{.partial} values are added to the beginning to shif the series to the Right. This is common in
Financial Applications such as moving average cross-overs.
}

\strong{Partial Values}
\itemize{
\item The advantage to using \code{.partial} values vs \code{NA} padding is that
the series can be filled (good for time-series de-noising operations).
\item The downside to partial values is that the partials can become less stable
at the regions where incomplete windows are used.
}

If instability is not desirable for de-noising operations, a suitable alternative
is \code{\link[=smooth_vec]{smooth_vec()}}, which implements local polynomial regression.
}
\examples{
library(dplyr)
library(ggplot2)

# Training Data
FB_tbl <- FANG \%>\%
    filter(symbol == "FB") \%>\%
    select(symbol, date, adjusted)

# ---- FUNCTION FORMAT ----
# - The `.f = mean` function is used. Argument `na.rm = TRUE` is passed as ...
FB_tbl \%>\%
    mutate(adjusted_30_ma = slidify_vec(
        .x      = adjusted,
        .period = 30,
        .f      = mean,
        na.rm   = TRUE,
        .align  = "center")) \%>\%
        ggplot(aes(date, adjusted)) +
        geom_line() +
        geom_line(aes(y = adjusted_30_ma), color = "blue", na.rm = TRUE)

# ---- FORMULA FORMAT ----
# - Anonymous function `.f = ~ mean(., na.rm = TRUE)` is used
FB_tbl \%>\%
    mutate(adjusted_30_ma = slidify_vec(
        .x      = adjusted,
        .period = 30,
        .f      = ~ mean(., na.rm = TRUE),
        .align  = "center")) \%>\%
        ggplot(aes(date, adjusted)) +
        geom_line() +
        geom_line(aes(y = adjusted_30_ma), color = "blue", na.rm = TRUE)

# ---- PARTIAL VALUES ----
# - set `.partial = TRUE`
FB_tbl \%>\%
    mutate(adjusted_30_ma = slidify_vec(
        .x       = adjusted,
        .f       = ~ mean(., na.rm = TRUE),
        .period  = 30,
        .align   = "center",
        .partial = TRUE)) \%>\%
        ggplot(aes(date, adjusted)) +
        geom_line() +
        geom_line(aes(y = adjusted_30_ma), color = "blue")

# ---- Loess vs Moving Average ----
# - Loess: Using `.degree = 0` to make less flexible. Comparable to a moving average.

FB_tbl \%>\%
    mutate(
        adjusted_loess_30 = smooth_vec(adjusted, period = 30, degree = 0),
        adjusted_ma_30    = slidify_vec(adjusted, .f = mean,
                                           .period = 30, .partial = TRUE)
    ) \%>\%
    ggplot(aes(date, adjusted)) +
    geom_line() +
    geom_line(aes(y = adjusted_loess_30), color = "red") +
    geom_line(aes(y = adjusted_ma_30), color = "blue") +
    labs(title = "Loess vs Moving Average")



}
\references{
\itemize{
\item \href{https://slider.r-lib.org/}{Slider R Package} by Davis Vaughan
}
}
\seealso{
Modeling and More Complex Rolling Operations:
\itemize{
\item \code{\link[=step_slidify]{step_slidify()}} - Roll apply for \code{tidymodels} modeling
\item \code{\link[=tk_augment_slidify]{tk_augment_slidify()}} - Add many rolling columns group-wise
\item \code{\link[=slidify]{slidify()}} - Turn any function into a rolling function. Great for
rolling cor, rolling regression, etc.
\item For more complex rolling operations, check out the \code{slider} R package.
}

Vectorized Transformation Functions:
\itemize{
\item Box Cox Transformation: \code{\link[=box_cox_vec]{box_cox_vec()}}
\item Lag Transformation: \code{\link[=lag_vec]{lag_vec()}}
\item Differencing Transformation: \code{\link[=diff_vec]{diff_vec()}}
\item Rolling Window Transformation: \code{\link[=slidify_vec]{slidify_vec()}}
\item Loess Smoothing Transformation: \code{\link[=smooth_vec]{smooth_vec()}}
\item Fourier Series: \code{\link[=fourier_vec]{fourier_vec()}}
\item Missing Value Imputation for Time Series: \code{\link[=ts_impute_vec]{ts_impute_vec()}}
}
}
