% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/classification.R
\name{reg_aucroc}
\alias{reg_aucroc}
\title{Area under the ROC curve for regression target outcomes}
\usage{
reg_aucroc(
  actual,
  pred,
  num_quants = 100,
  ...,
  cuts = NULL,
  imbalance = 0.05,
  na.rm = FALSE,
  sample_size = 10000,
  seed = 0
)
}
\arguments{
\item{actual}{numeric vector. Actual label values from a dataset. They must be numeric.}

\item{pred}{numeric vector. Predictions corresponding to each respective element in \code{actual}.}

\item{num_quants}{scalar positive integer. If \code{cuts} is \code{NULL} (default), \code{actual} will be dichotomized into \code{quants} quantiles and that many ROCs will be returned in the \code{rocs} element. However, if \code{cuts} is specified, then \code{quants} is ignored.}

\item{...}{Not used. Forces explicit naming of the arguments that follow.}

\item{cuts}{numeric vector. If \code{cuts} is provided, it overrides \code{quants} to specify the cut points for dichotomization of \code{actual} for the creation of \code{cuts + 1} ROCs.}

\item{imbalance}{numeric(1) in (0, 0.5]. The result element \code{mean_auc} averages the AUCs over three regions (see details of the return value). \code{imbalance} is the supposed percentage of the less frequent class in the data. If not provided, defaults to 0.05 (5\%).}

\item{na.rm}{See documentation for \code{aucroc()}}

\item{sample_size}{See documentation for \code{aucroc()}. In addition to those notes, for \code{reg_aucroc()}, any sampling is conducted before the dichotomization of \code{actual} so that all classification ROCs are based on identical data.}

\item{seed}{See documentation for \code{aucroc()}}
}
\value{
List with the following elements:
\itemize{
\item \code{rocs}: List of results for \code{aucroc()} for each dichotomized segment of \code{actual}.
\item \code{auc}: named numeric vector of AUC extracted from each element of \code{rocs}. Named by the percentile that the AUC represents.
\item \code{mean_auc}: named numeric(3). The average AUC over the low, middle, and high quantiles of dichotomization:
\item \code{lo}: average AUC with \code{imbalance}\% (e.g., 5\%) or less of the actual target values;
\item \code{mid}: average AUC in between \code{lo} and \code{hi};
\item \code{hi}: average AUC with (1 - \code{imbalance})\% (e.g., 95\%) or more of the actual target values;
}
}
\description{
Area under the ROC curve (AUCROC) is a classification measure. By dichotomizing the range of \code{actual} values, \code{reg_aucroc()} turns regression evaluation into classification evaluation for any regression model. Note that the model that generates the predictions is assumed to be a regression model; however, any numeric inputs are allowed for the \code{pred} argument, so there is no check for the nature of the source model.
}
\details{
The ROC data and AUCROC values are calculated with \code{aucroc()}.
}
\examples{
# Remove rows with missing values from airquality dataset
airq <- airquality |>
  na.omit()

# Create binary version where the target variable 'Ozone' is dichotomized based on its median
airq_bin <- airq
airq_bin$Ozone <- airq_bin$Ozone >= median(airq_bin$Ozone)

# Create a generic regression model; use autogam
req_aq   <- autogam::autogam(airq, 'Ozone', family = gaussian())
req_aq$perf$sa_wmae_mad  # Standardized accuracy for regression

# Create a generic classification model; use autogam
class_aq <- autogam::autogam(airq_bin, 'Ozone', family = binomial())
class_aq$perf$auc  # AUC (standardized accuracy for classification)

# Compute AUC for regression predictions
reg_auc_aq <- reg_aucroc(
  airq$Ozone,
  predict(req_aq)
)

# Average AUC over the lo, mid, and hi quantiles of dichotomization:
reg_auc_aq$mean_auc


}
