#' Plot skytrackr results
#'
#' Create a map of estimated locations as a static or dynamic map.
#'
#' @param df A data frame with locations produced with the skytrackr() function
#' @param bbox A geographic bounding box provided as a vector with the format
#'  xmin, ymin, xmax, ymax.
#' @param start_location A start location as lat/lon to indicate
#'  the starting position of the track (optional)
#' @param roi A region of interest under consideration, only used in
#'  plots during optimization
#' @param simplify simplify the plot and only show the map (default = FALSE)
#' @param intervals show the uncertainty intervals as shaded
#'  ellipses (default = FALSE)
#' @param dynamic Option to create a dynamic interactive graph rather than
#'  a static plot. Both the path as the locations are shown. The size
#'  of the points is proportional to the latitudinal uncertainty, while
#'  equinox windows are marked with red points. (default = FALSE)

#' @importFrom rlang .data

#' @return A ggplot map of tracked locations or mapview dynamic overview.
#' @export
#' @examples
#' \donttest{
#'
#' # define land mask with a bounding box
#' # and an off-shore buffer (in km), in addition
#' # you can specify the resolution of the resulting raster
#' mask <- stk_mask(
#'   bbox  =  c(-20, -40, 60, 60), #xmin, ymin, xmax, ymax
#'   buffer = 150, # in km
#'   resolution = 0.5 # map grid in degrees
#'   )
#'
#' # define a step selection distribution/function
#' ssf <- function(x, shape = 0.9, scale = 100, tolerance = 1500){
#'   norm <- sum(stats::dgamma(1:tolerance, shape = shape, scale = scale))
#'   prob <- stats::dgamma(x, shape = shape, scale = scale) / norm
#' }
#'
#' # estimate locations
#' locations <- cc876 |> skytrackr(
#'   plot = TRUE,
#'   mask = mask,
#'   step_selection = ssf,
#'   start_location = c(50, 4),
#'       control = list(
#'         sampler = 'DEzs',
#'         settings = list(
#'         iterations = 10, # change iterations
#'          message = FALSE
#'         )
#'       )
#'   )
#'
#' #----- actual plotting routines ----
#' # static plot, with required bounding box
#' locations |> stk_map()
#' }

stk_map <- function(
    df,
    bbox,
    start_location,
    roi,
    dynamic = FALSE,
    intervals = FALSE,
    simplify = FALSE
    ) {

   # check if not a {multidplyr} data frame
   if(inherits(df, "multidplyr_party_df")){
      df <- as.data.frame(df)
      cli::cli_alert(c(
         "Provided data is a {{multidplyr}} data frame.",
         "x" = "
           The data was converted to a standard data frame using:
           df <- as.data.frame(df)
         "
         )
      )
   }

   # show dynamic plot
   if(dynamic){

      # if not logger ID column exists
      # assign one for further processing
      if(! "logger" %in% colnames(df)){
        df$logger <- "Logger"
      }

      df <- df |>
         dplyr::mutate(
            uncertainty = .data$latitude_qt_95 - .data$latitude_qt_5
         ) |>
         dplyr::select(
            "logger",
            "longitude",
            "latitude",
            "sky_conditions",
            "grd",
            "date",
            "equinox",
            "grd"
         )

      points <- df |>
         dplyr::group_by(.data$logger) |>
         sf::st_as_sf(
            coords = c("longitude","latitude"),
            crs = 4326
         )

      path <-  points |>
         dplyr::group_by(.data$logger) |>
         dplyr::summarise(do_union = FALSE) |>
         sf::st_cast("MULTILINESTRING")

      m <- mapview::mapview(
         path,
         map.types = c("Esri.WorldImagery","OpenTopoMap")
      )

      m <- m + mapview::mapview(
         points,
         popup = TRUE,
         zcol = "equinox",
         col.regions = c("red", "white"),
         col = c("red", "white"),
         cex = "grd",
         label = NA
      )

      print(m)
      return(invisible())
   }

   # calculate convergence labels
   df <- df |>
      dplyr::mutate(
         convergence = ifelse(.data$grd < 1.2, "good","poor"),
         convergence = ifelse(.data$grd < 1.05, "best", .data$convergence)
      )

   df$convergence <- factor(
      df$convergence, levels=c("best", "good", "poor")
   )

   if (intervals){
      # create uncertainty intervals
      uncertainty <- suppressWarnings({suppressMessages({
         df |>
            sf::st_as_sf(
               coords = c("longitude_qt_50", "latitude_qt_50")
            ) |>
            sf::st_set_crs("EPSG:4326") |>
            dplyr::rowwise() |>
            dplyr::mutate(
               geometry = sfdep::st_ellipse(
                  .data$geometry,
                  sx = (.data$longitude_qt_95 - .data$longitude_qt_5)/2,
                  sy = (.data$latitude_qt_95 - .data$latitude_qt_5)/2
               )
            ) |>
            sf::st_geometry() |>
            sf::st_cast("POLYGON") |>
            sf::st_union() |>
            sf::st_transform(crs = "+proj=eqearth")
      })})
   }


   # convert to sf
   path <-  df |>
            sf::st_as_sf(coords = c("longitude", "latitude")) |>
            sf::st_set_crs("EPSG:4326") |>
            sf::st_combine() |>
            sf::st_cast("LINESTRING") |>
            sf::st_transform(crs = "+proj=eqearth")

   points <- sf::st_as_sf(df, coords = c("longitude", "latitude")) |>
      sf::st_set_crs("EPSG:4326") |>
      sf::st_cast("POINT") |>
      sf::st_transform(crs = "+proj=eqearth")

   # automatically calculate bounding box
   if(missing(bbox)){
      cli::cli_alert(c(
         "No bounding box (bbox) provided.",
         "x" = "
           Bounding box is estimated from the data.
         "
         )
      )

      # estimate bounding box
      bbox <- df |>
         dplyr::ungroup() |>
         dplyr::summarize(
            xmin = min(.data$longitude, na.rm = TRUE) - 5,
            ymin = min(.data$latitude, na.rm = TRUE) - 5,
            xmax = max(.data$longitude, na.rm = TRUE) + 5,
            ymax = max(.data$latitude, na.rm = TRUE) + 5
         ) |> unlist()
   }

   # mask polygon / crop to broad bounding box
   # region of interest
   m <- stk_mask(
      bbox = bbox,
      sf = TRUE
   )

   # transform to equal earth
   m <- m |>
      sf::st_transform(crs = "+proj=eqearth")

   # crop and convert rivers
   rivers <- suppressWarnings({suppressMessages({ rivers |>
      sf::st_crop(bbox) |>
      sf::st_transform(crs = "+proj=eqearth")
   })})

   # crop and convert lakes
   lakes <- suppressWarnings({suppressMessages({ lakes |>
      sf::st_crop(bbox) |>
      sf::st_transform(crs = "+proj=eqearth")
   })})

   # load raster
   r <- terra::rast(
      system.file("extdata/shadedrelief.tif", package = "skytrackr")
   ) |>
      terra::project("+proj=eqearth") |>
      terra::crop(terra::vect(m)) |>
      terra::mask(terra::vect(m))

   # base plot call
   p <- ggplot2::ggplot(df) +
      tidyterra::geom_spatraster_rgb(
         data = r
      )

   if(missing(roi)){
      p <- p +
         ggplot2::geom_sf(
            data = rivers,
            fill = NA,
            colour = "lightblue",
            alpha = 0.6,
            inherit.aes = FALSE,
            na.rm = TRUE
         ) +
         ggplot2::geom_sf(
            data = lakes,
            fill = "lightblue",
            colour = NA,
            alpha = 0.6,
            inherit.aes = FALSE,
            na.rm = TRUE
         ) +
         ggplot2::geom_sf(
            data = m,
            fill = NA,
            colour = "black",
            inherit.aes = FALSE,
            na.rm = TRUE
         ) +
         ggplot2::theme_bw() +
         ggplot2::theme(
            legend.position = "bottom",
            panel.border = ggplot2::element_blank(),
            plot.margin = ggplot2::margin(
               t = 10,
               r = 0,
               b = 0,
               l = 0,
               unit = "pt")
         )
   }

   if(!missing(roi)){

      # intersection of roi with mask
      roi <-   m |>  sf::st_intersection(
         roi |> sf::st_transform(crs = "+proj=eqearth")
      )

     p <- p +
        ggplot2::geom_sf(
           data = rivers,
           fill = NA,
           colour = "lightblue",
           alpha = 0.6,
           inherit.aes = FALSE,
           na.rm = TRUE
        ) +
        ggplot2::geom_sf(
           data = lakes,
           fill = "lightblue",
           colour = NA,
           alpha = 0.6,
           inherit.aes = FALSE,
           na.rm = TRUE
        ) +
        ggplot2::geom_sf(
           data = roi,
           colour = "red",
           fill = NA,
           lty = 2,
           alpha = 0.5,
           na.rm = TRUE
        ) +
        ggplot2::geom_sf(
           data = m,
           colour = "black",
           fill = NA,
           na.rm = TRUE
        ) +
        ggplot2::theme_bw() +
        ggplot2::theme(
           legend.position = "bottom",
           panel.border = ggplot2::element_blank(),
           plot.margin = ggplot2::margin(
              t = 10,
              r = 0,
              b = 0,
              l = 0,
              unit = "pt")
        )
   }

   if(nrow(df) >= 1) {

      if(intervals){
         p <- p +
            ggplot2::geom_sf(
            data = uncertainty,
            colour = NA,
            fill = "grey25",
            alpha = 0.3,
            na.rm = TRUE
         )
      }

      p <- p +
         ggplot2::geom_sf(
            data = path,
            colour = "grey25",
            lty = 3,
            fill = NA,
            na.rm = TRUE
         ) +
         ggplot2::geom_sf(
            data = points,
            ggplot2::aes(
               shape = .data$equinox,
               colour = .data$convergence,
               fill = .data$convergence,
            ),
            na.rm = TRUE
         ) +
         ggplot2::scale_shape_manual(
            values = c(20, 3)
          ) +
         ggplot2::scale_colour_manual(
            values = c("poor" = "#d7191c","good" = "#ffffbf", "best" = "#2c7bb6"),
            name = "fit"
         ) +
         ggplot2::scale_fill_manual(
            values = c("poor" = "#d7191c","good" = "#ffffbf", "best" = "#2c7bb6"),
            guide = "none"
         ) +
         ggplot2::geom_sf(
            data = points |> dplyr::filter(date == date[nrow(points)]),
            colour = "black",
            pch = 1,
            size = 4,
            na.rm = TRUE
         )
   }

   if(!missing(start_location)){
      start <- sf::st_as_sf(
         data.frame(
            latitude = start_location[1],
            longitude = start_location[2]
         ),
         coords = c("longitude", "latitude")
      ) |>
         sf::st_set_crs("EPSG:4326") |>
         sf::st_cast("POINT") |>
         sf::st_transform(crs = "+proj=eqearth")

      p <- p +ggplot2::geom_sf(
         data = start,
         colour = "black",
         pch = 17,
         size = 3,
         na.rm = TRUE
      )
   }

   # return the simple map plot
   if(simplify){
      return(p)
   }

   p_lat <- ggplot2::ggplot(df) +
     ggplot2::geom_ribbon(
       ggplot2::aes(
         y = .data$date,
         xmin = .data$latitude_qt_5,
         xmax = .data$latitude_qt_95,
         group = .data$logger
       ),
       fill = "grey85"
     ) +
     ggplot2::geom_path(
       ggplot2::aes(
         y = .data$date,
         x = .data$latitude_qt_50,
         group = .data$logger
       ),
       colour = "grey50"
     )  +
      ggplot2::geom_path(
         ggplot2::aes(
            y = .data$date,
            x = .data$latitude,
            group = .data$logger
         ),
         colour = "black"
      )  +
      ggplot2::labs(
         x = "latitude"
      ) +
     ggplot2::theme_bw()

   p_lon <- ggplot2::ggplot(df) +
     ggplot2::geom_ribbon(
       ggplot2::aes(
         y = .data$date,
         xmin = .data$longitude_qt_5,
         xmax = .data$longitude_qt_95,
         group = .data$logger
       ),
       fill = "grey85"
     ) +
     ggplot2::geom_path(
       ggplot2::aes(
         y = .data$date,
         x = .data$longitude_qt_50,
         group = .data$logger
       ),
       colour = "grey50"
     )  +
      ggplot2::geom_path(
         ggplot2::aes(
            y = .data$date,
            x = .data$longitude,
            group = .data$logger
         ),
         colour = "black"
      )  +
      ggplot2::labs(
         x = "longitude"
      ) +
     ggplot2::theme_bw()

   p_sky <- ggplot2::ggplot(df) +
      ggplot2::geom_ribbon(
         ggplot2::aes(
            y = .data$date,
            xmin = .data$sky_conditions_qt_5,
            xmax = .data$sky_conditions_qt_95,
            group = .data$logger
         ),
         fill = "grey85"
      ) +
     ggplot2::geom_path(
       ggplot2::aes(
         y = .data$date,
         x = .data$sky_conditions_qt_50,
         group = .data$logger
       ),
       colour = "grey50"
     ) +
      ggplot2::geom_path(
         ggplot2::aes(
            y = .data$date,
            x = .data$sky_conditions,
            group = .data$logger
         ),
         colour = "black"
      )  +
     ggplot2::labs(
        x = "sky conditions"
     ) +
     ggplot2::theme_bw()

   p_grd <- ggplot2::ggplot(df) +
      ggplot2::geom_point(
         ggplot2::aes(
            y = .data$date,
            x = .data$grd,
            group = .data$logger
         ),
         colour = "grey50",
         na.rm = TRUE
      ) +
      ggplot2::geom_vline(xintercept = 1.2) +
      ggplot2::geom_vline(xintercept = 1.05, linetype = 2) +
      ggplot2::labs(
         x = "Gelman-Rubin\n diagnostic"
      ) +
      ggplot2::theme_bw() +
      ggplot2::theme(
         legend.position = "bottom"
      )

   p_final <- p + p_lat + p_lon + p_sky + p_grd +
     patchwork::plot_layout(
       ncol = 5,
       widths = c(4, 1, 1, 1, 1),
       axis_titles = "collect_y"
      )

   return(p_final)
}
