% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/elementary_index.R
\name{elementary_index}
\alias{elementary_index}
\alias{elementary_index.default}
\alias{elementary_index.numeric}
\alias{elementary_index.data.frame}
\alias{elemental_index}
\title{Make elementary/elemental price indexes}
\usage{
elementary_index(x, ...)

\method{elementary_index}{default}(x, ...)

\method{elementary_index}{numeric}(
  x,
  ...,
  period = gl(1, length(x)),
  ea = gl(1, length(x)),
  weights = NULL,
  product = NULL,
  chainable = TRUE,
  na.rm = FALSE,
  contrib = FALSE,
  r = 0
)

\method{elementary_index}{data.frame}(x, formula, ..., weights = NULL, product = NULL)

elemental_index(x, ...)
}
\arguments{
\item{x}{Period-over-period or fixed-base price relatives. Currently there
are methods for numeric vectors (which can be made with
\code{\link[=price_relative]{price_relative()}}) and data frames.}

\item{...}{Further arguments passed to or used by methods.}

\item{period}{A factor, or something that can be coerced into one, giving
the time period associated with each price relative in \code{x}. The
ordering of time periods follows of the levels of \code{period}, to agree
with \code{\link[=cut.Date]{cut()}}. The default makes an index for one time period.}

\item{ea}{A factor, or something that can be coerced into one, giving the
elementary aggregate associated with each price relative in \code{x}. The
default makes an index for one elementary aggregate.}

\item{weights}{A numeric vector of weights for the price relatives in \code{x},
or something that can be coerced into one. The default is equal weights.
This is evaluated in \code{x} for the data frame method.}

\item{product}{A character vector of product names, or something that can
be coerced into one, for each price relative in \code{x} when making
percent-change contributions. The default uses the names of \code{x}, if any;
otherwise, elements of \code{x} are given sequential names within each
elementary aggregate. This is evaluated in \code{x} for the data frame method.}

\item{chainable}{Are the price relatives in \code{x} period-over-period
relatives that are suitable for a chained calculation (the default)? This
should be \code{FALSE} when \code{x} contains fixed-base relatives.}

\item{na.rm}{Should missing values be removed? By default, missing values
are not removed. Setting \code{na.rm = TRUE} is equivalent to overall mean
imputation.}

\item{contrib}{Should percent-change contributions be calculated? The
default does not calculate contributions.}

\item{r}{Order of the generalized mean to aggregate price relatives. 0 for a
geometric index (the default for making elementary indexes), 1 for an
arithmetic index (the default for aggregating elementary indexes and
averaging indexes over subperiods), or -1 for a harmonic index (usually for
a Paasche index). Other values are possible; see
\code{\link[gpindex:generalized_mean]{gpindex::generalized_mean()}} for details.}

\item{formula}{A two-sided formula with price relatives on the left-hand
side, and time periods and elementary aggregates (in that order) on the
right-hand side.}
}
\value{
A price index that inherits from \code{\link{piar_index}}. If
\code{chainable = TRUE} then this is a period-over-period index that also
inherits from \code{\link{chainable_piar_index}}; otherwise, it is a
fixed-based index that inherits from \code{\link{direct_piar_index}}.
}
\description{
Compute period-over-period (chainable) or fixed-base (direct) elementary
price indexes, with optional percent-change contributions for each
product.
}
\details{
When supplied with a numeric vector, \code{elementary_index()} is a simple
wrapper that applies
\code{\link[gpindex:generalized_mean]{gpindex::generalized_mean(r)()}} and
\code{\link[gpindex:contributions]{gpindex::contributions(r)()}} (if \code{contrib = TRUE})
to \code{x} and \code{weights} grouped by \code{ea} and \code{period}. That
is, for every combination of elementary aggregate and time period,
\code{elementary_index()} calculates an index based on a generalized mean of
order \code{r} and, optionally, percent-change contributions. Product names should
be unique within each time period when making contributions, and, if not, are
passed to \code{\link[=make.unique]{make.unique()}} with a warning. The default
(\code{r = 0} and no weights) makes Jevons elementary indexes. See chapter 8
(pp. 175--190) of the CPI manual (2020) for more detail about making
elementary indexes, or chapter 9 of the PPI manual (2004), and chapter 5 of
Balk (2008).

The default method simply coerces \code{x} to a numeric vector prior to
calling the method above. The data frame method provides a formula interface
to specify columns of price relatives, time periods, and elementary
aggregates and call the method above.

The interpretation of the index depends on how the price relatives in
\code{x} are made. If these are period-over-period relatives, then the
result is a collection of period-over-period (chainable) elementary indexes;
if these are fixed-base relatives, then the result is a collection of
fixed-base (direct) elementary indexes. For the latter, \code{chainable}
should be set to \code{FALSE} so that no subsequent methods assume that a
chained calculation should be used.

By default, missing price relatives in \code{x} will propagate throughout
the index calculation. Ignoring missing values with \code{na.rm = TRUE} is
the same as overall mean (parental) imputation, and needs to be explicitly
set in the call to \code{elementary_index()}. Explicit imputation of missing
relatives, and especially imputation of missing prices, should be done prior
to calling \code{elementary_index()}.

Indexes based on nested generalized means, like the Fisher index (and
superlative quadratic mean indexes more generally), can be calculated by
supplying the appropriate weights with \code{\link[gpindex:transmute_weights]{gpindex::nested_transmute()}}; see the
example below. It is important to note that there are several ways to
make these weights, and this affects how percent-change contributions
are calculated.

\code{elemental_index()} is an alias for \code{elementary_index()}.
}
\examples{
library(gpindex)

prices <- data.frame(
  rel = 1:8,
  period = rep(1:2, each = 4),
  ea = rep(letters[1:2], 4)
)

# Calculate Jevons elementary indexes

elementary_index(prices, rel ~ period + ea)

# Same as using lm() or tapply()

exp(coef(lm(log(rel) ~ ea:factor(period) - 1, prices)))

with(
  prices,
  t(tapply(rel, list(period, ea), geometric_mean, na.rm = TRUE))
)

# A general function to calculate weights to turn the geometric
# mean of the arithmetic and harmonic mean (i.e., Fisher mean)
# into an arithmetic mean

fw <- grouped(nested_transmute(0, c(1, -1), 1))

# Calculate a CSWD index (same as the Jevons in this example)
# as an arithmetic index by using the appropriate weights

elementary_index(
  prices,
  rel ~ period + ea,
  weights = fw(rel, group = interaction(period, ea)),
  r = 1
)

}
\references{
Balk, B. M. (2008). \emph{Price and Quantity Index Numbers}.
Cambridge University Press.

ILO, IMF, UNECE, OECD, and World Bank. (2004).
\emph{Producer Price Index Manual: Theory and Practice}.
International Monetary Fund.

IMF, ILO, OECD, Eurostat, UNECE, and World Bank. (2020).
\emph{Consumer Price Index Manual: Concepts and Methods}.
International Monetary Fund.

von der Lippe, P. (2007). \emph{Index Theory and Price Statistics}. Peter Lang.
}
\seealso{
\code{\link[=price_relative]{price_relative()}} for making price relatives for the same products over
time, and \code{\link[=carry_forward]{carry_forward()}} and \code{\link[=shadow_price]{shadow_price()}} for
imputation of missing prices.

\code{\link[=as_index]{as_index()}} to turn pre-computed (elementary) index values into an
index object.

\code{\link[=chain]{chain()}} for chaining period-over-period indexes, and
\code{\link[=rebase]{rebase()}} for rebasing an index.

\code{\link[=aggregate.piar_index]{aggregate()}} to aggregate elementary indexes
according to an aggregation structure.

\code{\link[=as.matrix.piar_index]{as.matrix()}} and
\code{\link[=as.data.frame.piar_index]{as.data.frame()}} for coercing an index
into a tabular form.
}
