#' Droop formula
#'
#' @rdname ElecFuns
#' 
#' @section Details: The droop function assigns seats by calculating the Droop 
#' quota, which is Q = V/(M+1) + 1 (rounded to the nearest integer). The seats 
#' are assigned in two stages. First, the number of votes obtained by each 
#' party is divided by Q and rounded down. That integer is the number of seats 
#' that the party will obtain in the first stage. Second, Q is multiplied by 
#' the number of seats obtained by each party (Si) and that number if 
#' substracted from the total number of votes obtained by that party. That would 
#' be a residual: Ri = Vi - Q*Si. The remaining seats are assigned to the 
#' parties with the largest residuals.  
#' 
#' @export
#'
#' @examples 
#' 
#' ## Droop without threshold: 
#' 
#' droop(v=example, m=3)
#' 
#' ## Droop with 20% threshold: 
#' 
#' droop(v=example, m=3, threshold=0.2) 
droop <- function(v,
                  m,
                  threshold = 0.0, 
                  ...){
  # this snippet uses hare with largest remainders (as opposed to average remainders)
  # m is district magnitude
  if(nrow(v) > 1){
    stop("Droop undefined for ranked votes.")
  }
  if(threshold > max(v)/sum(v)){
    stop("Threshold is higher than maximum proportion of votes")
  }
  findSeats <- function (x, threshold = threshold) {
    # apply threshold
    if(sum(prop.table(x[1, ]) < threshold) != ncol(x)){
      x[1, prop.table(x[1, ]) < threshold]  <- 0         
    }
    quota <- floor(sum(x)/(m + 1)) + 1 # droop quota is floor(total votes/(m + 1)) + 1
    quota.integer <-  x %/% quota  # Seats immediately awarded
    quota.remainder <- x %% quota # largest remainders
    remaining.seats <- m - sum(quota.integer) # total seats awarded to remainders
    if(remaining.seats < 0){ # if sum(quota.integer) > m
      while(remaining.seats < 0){
        quota.integer[1, which.min(quota.remainder)] <- quota.integer[1, which.min(quota.remainder)] - 1
        quota.remainder[which.min(quota.remainder)] <- quota.remainder[which.min(quota.remainder)] + quota
        remaining.seats <- remaining.seats + 1
      }
    } else{
      extra.seats <- 0
    }    
    if(remaining.seats > 0){
      remaining.seats.winners <- order(-quota.remainder)[1:remaining.seats] # winners of remainders
      extra.seats <- rep (0, length(x))
      extra.seats[remaining.seats.winners] <- 1
    } else {
      extra.seats <- 0
    }
    seats <- c(quota.integer + extra.seats)  # total seats
    return (as.numeric(seats)) # PC: Add as.numeric because it was returning party names
    # return (seats) # Old version
  }
  dist.seats <- findSeats(x = v, threshold = threshold)
  names(dist.seats) <- colnames(v)
  return (dist.seats)
}
