#' Get the optimal built-up area threshold
#'
#' @description
#' In Data Package 2023, the Degree of Urbanisation includes a "Reduce Fragmentation Option" to account for the presence of office parks, shopping malls, factories and transport infrastructure. When the setting is enabled, urban centres are created using both cells with a population density of at least 1500 inhabitants per km² *and* cells that have an "optimal" built-up area on permanent land. This function can be used to determine this "optimal" threshold value.
#'
#' The optimal threshold is dynamically identified as the global average built-up area proportion in clusters with a density of at least 1500 inhabitants per permanent land with a minimum population of 5000 people. We empirically discovered that the Degree of Urbanisation uses the rounded up (ceiled) optimal threshold to two decimal places.
#'
#' For more information: see [GHSL Data Package 2023, footnote 30](https://ghsl.jrc.ec.europa.eu/documents/GHSL_Data_Package_2023.pdf). The arguments of the function allow to adapt the standard specifications in order to determine an alternative "optimal" threshold.

#' @param  data path to the directory with the data, or named list with the data as returned by function [DoU_preprocess_grid()]. Ideally, it contains data on a global scale.
#' @param density_threshold numeric. Minimum population density per permanent land
#' @param size_threshold numeric. Minimum population size
#' @param directions integer. Which cells are considered adjacent: `4` for rooks case (horizontal and vertical neighbours) or `8` for queens case (horizontal, vertical and diagonal neighbours)
#' @return optimal built-up area threshold
#' @examples
#  # As illustration, the function is applied for Belgium, ideally the function should be applied at a global scale
#' data_belgium <- DoU_load_grid_data_belgium()
#'
#' # determine the optimal built-up threshold with standard specifications
#' DoU_get_optimal_builtup(data_belgium)
#'
#' # determine the optimal built-up threshold with custom specification
#' DoU_get_optimal_builtup(data_belgium,
#'   density_threshold = 1000,
#'   size_threshold = 3500,
#'   directions = 8
#' )
#' @export
DoU_get_optimal_builtup <- function(
  data,
  density_threshold = 1500,
  size_threshold = 5000,
  directions = 4
) {
  # read data if data is character
  if (is.character(data)) {
    data <- DoU_preprocess_grid(data)
  }

  # check if data is valid
  if (
    !all(
      c("pop", "land", "pop_per_land", "built_per_land", "built") %in%
        names(data)
    )
  ) {
    stop(
      "Invalid argument: The data should contain a named list of pop, land, built, pop_per_land and built_per_land as generated by the function DoU_preprocess_grid."
    )
  }

  # if a buffer was applied: remove the buffer to calculate the optimal threshold
  if (!is.null(data$metadata_POP$buffer) && (data$metadata_POP$buffer > 0)) {
    pop_per_land <- terra::crop(
      data$pop_per_land,
      terra::ext(data$pop_per_land) - data$metadata_POP$buffer
    )
    pop <- terra::crop(
      data$pop,
      terra::ext(data$pop) - data$metadata_POP$buffer
    )
    built <- terra::crop(
      data$built,
      terra::ext(data$built) - data$metadata_POP$buffer
    )
  } else {
    pop_per_land <- data$pop_per_land
    pop <- data$pop
    built <- data$built
  }

  # get dense patches
  densepatches <- flexurba::get_clusters(
    xden = pop_per_land,
    minden = density_threshold,
    xsiz = pop,
    minsiz = size_threshold,
    directions = directions
  )

  # mask with built-up layer
  masked_built <- terra::mask(built, densepatches)

  return(as.numeric(terra::global(masked_built, "mean", na.rm = TRUE)))
}

#' Preprocess the data for the DEGURBA spatial units classification
#'
#' @description
#' `r lifecycle::badge("deprecated")`
#'
#' `get_optimal_builtup()` has been renamed to `DoU_get_optimal_builtup()` to create a more consistent API and to better indicate that this function is specifically designed for detecting the optimal builtup threshold in the context of the DEGURBA classification with `DoU_classify_units()`.
#' @param  data path to the directory with the data, or named list with the data as returned by function [DoU_preprocess_grid()]. Ideally, it contains data on a global scale.
#' @param density_threshold numeric. Minimum population density per permanent land
#' @param size_threshold numeric. Minimum population size
#' @param directions integer. Which cells are considered adjacent: `4` for rooks case (horizontal and vertical neighbours) or `8` for queens case (horizontal, vertical and diagonal neighbours)
#' @return optimal built-up area threshold
#' @keywords internal
#' @export
get_optimal_builtup <- function(
  data,
  density_threshold = 1500,
  size_threshold = 5000,
  directions = 4
) {
  return(DoU_get_optimal_builtup(
    data,
    density_threshold,
    size_threshold,
    directions
  ))
}
