% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cat_cox.R
\name{cat_cox}
\alias{cat_cox}
\title{Catalytic Cox Proportional Hazards Model (COX) Fitting Function with Fixed Tau}
\usage{
cat_cox(
  formula,
  cat_init,
  tau = NULL,
  method = c("CRE", "WME"),
  init_coefficients = NULL,
  tol = 1e-05,
  max_iter = 25
)
}
\arguments{
\item{formula}{A formula specifying the Cox model. Should at least include response variables (e.g. \code{~ .}).}

\item{cat_init}{A list generated from \code{cat_cox_initialization}.}

\item{tau}{Optional numeric scalar controlling the weight of the synthetic data in the coefficient estimation, defaults to the number of predictors.}

\item{method}{The estimation method, either \code{"CRE"} (Catalytic-regularized Estimator) or \code{"WME"} (Weighted Mixture Estimator).}

\item{init_coefficients}{Initial coefficient values before iteration. Defaults to zero if not provided (if using \code{CRE}).}

\item{tol}{Convergence tolerance for iterative methods. Default is \code{1e-5} (if using \code{CRE}).}

\item{max_iter}{Maximum number of iterations allowed for convergence. Default is \code{25} (if using \code{CRE}).}
}
\value{
A list containing the values of all the arguments and the following components:
\item{coefficients}{Estimated coefficient vector.}
\item{model}{Fitted Cox model object (if using \code{WME}).}
\item{iteration_log}{Matrix logging variance and coefficient values for each iteration(if using \code{CRE}).}
\item{iter}{Number of iterations (if using \code{CRE}).}
}
\description{
Fits a Catalytic Cox proportional hazards model for survival data with specified variance parameters
and iterative coefficient estimation, with either \code{CRE} (Catalytic-regularized Estimator) or \code{WME} (Weighted Mixture Estimator) methods.
}
\examples{
library(survival)
data("cancer")
cancer$status[cancer$status == 1] <- 0
cancer$status[cancer$status == 2] <- 1

cat_init <- cat_cox_initialization(
  formula = Surv(time, status) ~ 1, # formula for simple model
  data = cancer,
  syn_size = 100, # Synthetic data size
  hazard_constant = 0.1, # Hazard rate value
  entry_points = rep(0, nrow(cancer)), # Entry points of each observation
  x_degree = rep(1, ncol(cancer) - 2), # Degrees for polynomial expansion of predictors
  resample_only = FALSE, # Whether to perform resampling only
  na_replace = stats::na.omit # How to handle NA values in data
)

cat_model_cre <- cat_cox(
  formula = ~.,
  cat_init = cat_init, # Only accept object generated from `cat_cox_initialization`
  tau = 1, # Weight for synthetic data
  method = "CRE", # Choose from `"CRE"` or `"WME"`
  init_coefficients = rep(0, ncol(cat_init$x)), # Initial coefficient values (Only for `CRE`)
  tol = 1e-1, # Tolerance for convergence criterion  (Only for `CRE`)
  max_iter = 3 # Maximum number of iterations for convergence  (Only for `CRE`)
)
cat_model_cre

cat_model_wme <- cat_cox(
  formula = ~.,
  cat_init = cat_init, # Only accept object generated from `cat_cox_initialization`
  tau = 1, # Weight for synthetic data
  method = "WME"
)
cat_model_wme
}
