% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/main.r
\name{simulate_network}
\alias{simulate_network}
\title{Simulate the failures, expansion, rehabilitation, and costs of a network}
\usage{
simulate_network(n.years, expansion, rehabilitation, prob.failure,
  income = 0, initial.budget = Inf, initial.inventory = NULL,
  free.expansion = TRUE)
}
\arguments{
\item{n.years}{number of years to simulate}

\item{expansion}{either a scalar describing the number of pipes added
every year to expand the pipe network, or a vector of length \code{n.years}.
Negative values are not allowed.}

\item{rehabilitation}{a (combination of) rehabilitation strategy function(s). See details below.}

\item{prob.failure}{a function describing the probability of a pipe failing in the next year
given its age, number of previous failures, and the age at the last failure (if any).}

\item{income}{either a scalar describing the annual income, or a vector of length \code{n.years}.}

\item{initial.budget}{initial budget}

\item{initial.inventory}{if it is an integer it specifies the
number of initial pipes, or alternatively it can be a \code{data.frame}
containing the initial inventory of pipes.}

\item{free.expansion}{if \code{TRUE} costs for network expansion are not deducted from the budget.}
}
\value{
an updated state list
}
\description{
Simulates failures, expansion, rehabilitation, and costs of a water supply pipe network.
The simulation is stochastic.
}
\details{
The rehabilitation is defined by combining different simple replacement strategies.
See the example for how this can be done using the \code{mystrategy} function input.
If the strategies vary over time, see \code{\link{initiate.network}} and
\code{\link{simulate_network.period}}.

The failure behavior is defined by the function \code{prob.failure}.
It calculates the probability of a pipe failing within the next year based on pipe age,
pipe age at the last failure, and the number of failures. Note, the model
makes the assumption that a pipe cannot fail more than once per year.

The costs are calculated as a function of the pipe diameter, assuming all pipes have a
length of 100 meters.
}
\examples{
## -----------
## define model parameters

## Define the annual probability of a failure
prob.failure.exp <- function(age, age.last.failure, n.failure) {
  if(n.failure==0){
    return(1/30)
  } else {
    return(1/10)
  }
}

## define a complicated (and probably useless) rehabilitation strategy
mystrategy <- . \%>\%
  replace.n.highest.risk(n=2, prob.failure=prob.failure.exp) \%>\%
  replace.more.failures.than(failures=5) \%>\%
  replace.older.than(age=70, max.cost=2e6)  \%>\%
  replace.n.oldest(n=3) \%>\%
  replace.n.random(n=2)
## This means: every year (if we have enough budget!), replace first the 2 pipes
## with the highest risk, then all pipes with more than 5 failures,
## then all pipes older then 70 years (up to costs of 2e6), then the 3
## oldest pipes remaining, and finally replace 2 randomly selected pipes. 


## -----------
## run the simulation

result <- simulate_network(

    n.years = 100,                   # run it for 100 years
    expansion = 10,                  # build 10 pipes per year (if money is available)
    rehabilitation = mystrategy,     # use the strategy defined above
    prob.failure = prob.failure.exp, # use the probability function defined above
    income = 1e6,                    # the annual income
    initial.budget = 1e7,   
    initial.inventory = 50,          # start the simulation with 50 new pipes
    free.expansion = FALSE
     
     )          

## look at some results
## str(result)
## str(result$time.100)

}
\seealso{
For more fine-grained control see \code{\link{initiate.network}}
and \code{\link{simulate_network.period}}. Different replacement strategies
are implemented: \code{\link{replace.n.highest.risk}},
\code{\link{replace.n.oldest}}, \code{\link{replace.n.random}}, \code{\link{replace.older.than}},
\code{\link{replace.more.failures.than}}, \code{\link{do.nothing}}.
}
\author{
Andreas Scheidegger
}
