#' Import a download requested from GBIF
#'
#' @description
#' This function imports a dataset downloaded from GBIF using a request key
#' generated by the `request_gbif()` function. It optionally allows saving
#' the imported occurrences to disk in CSV or GZIP format.
#'
#' @param request_key an object of class 'request_key' returned by the
#' `request_gbif()` function.
#' @param write_file whether to save the downloaded occurrences to disk.
#' Default is FALSE. If TRUE, you must specify the `output_dir`.
#' @param output_dir (character) a directory to save the data downloaded from
#' GBIF. Only applicable if `write_file = TRUE`. Default is NULL.
#' @param file.format (character) the format to save the file. Options available
#' are 'csv' (comma-separated values) and 'gz' (compressed GZIP). Only
#' applicable if `write_file = TRUE`. Default is 'gz'.
#' @param select_columns (logical) whether to import only specific columns (TRUE)
#' or all columns (FALSE) from the occurrence table. Default is TRUE.
#' @param columns_to_import (character) vector of column names to import.
#' Default is NULL, meaning it will import the column names specified in
#' `RuHere::gbif_columns` data.
#' @param overwrite (logical) whether to overwrite the file in the 'output_dir'
#' if it already exists. Default is FALSE.
#' @param ... other arguments passed to `rgbif::occ_download_import()`.
#'
#' @returns
#' A data frame containing the GBIF occurrence records. If `write_file = TRUE`,
#' the function also saves the dataset to disk in the specified format.
#'
#' @note
#' This function requires an active internet connection.
#'
#' @importFrom rgbif occ_download_get occ_download_import
#' @importFrom data.table fwrite
#'
#' @export
#'
#' @examples
#'  \dontrun{
#' # Prepare data to request GBIF download
#' gbif_prepared <- prepare_gbif_download(species = "Araucaria angustifolia")
#' # Submit a request to download occurrences
#' gbif_requested <- request_gbif(gbif_info = gbif_prepared)
#' # Check progress
#' rgbif::occ_download_wait(gbif_requested)
#' # After succeeded, import data
#' occ_gbif <- import_gbif(request_key = gbif_requested)
#' }
import_gbif <- function(request_key,
                        write_file = FALSE,
                        output_dir = NULL,
                        file.format = "gz",
                        select_columns = TRUE,
                        columns_to_import = NULL,
                        overwrite = FALSE,
                        ...){

  # Check arguments
  # 'request_key' must be of class 'request_key'
  if (!inherits(request_key, "request_key"))
    stop("'request_key' must be an object of class 'request_key', not ",
         paste(class(request_key), collapse = ", "), call. = FALSE)

  # 'write_file' (logical)
  if (!inherits(write_file, "logical") || length(write_file) != 1)
    stop("'write_file' must be a single logical value (TRUE or FALSE).", call. = FALSE)

  # 'output_dir' (character), required only if write_file = TRUE
  if (isTRUE(write_file)) {
    if (is.null(output_dir))
      stop("You must specify 'output_dir' when 'write_file = TRUE'.", call. = FALSE)
    if (!inherits(output_dir, "character") || length(output_dir) != 1)
      stop("'output_dir' must be a single character string.", call. = FALSE)
    if (!dir.exists(output_dir))
      stop("The directory specified in 'output_dir' does not exist.", call. = FALSE)
  }

  # 'file.format' (character)
  valid_formats <- c("csv", "gz")
  if (!inherits(file.format, "character") || length(file.format) != 1 ||
      !file.format %in% valid_formats)
    stop("'file.format' must be one of: ", paste(valid_formats, collapse = ", "), call. = FALSE)

  # 'select_columns' (logical)
  if (!inherits(select_columns, "logical") || length(select_columns) != 1)
    stop("'select_columns' must be a single logical value (TRUE or FALSE).", call. = FALSE)

  # 'columns_to_import' (character or NULL)
  if (!is.null(columns_to_import) && !inherits(columns_to_import, "character"))
    stop("'columns_to_import' must be a character vector or NULL.", call. = FALSE)

  # 'overwrite' (logical)
  if (!inherits(overwrite, "logical") || length(overwrite) != 1)
    stop("'overwrite' must be a single logical value (TRUE or FALSE).", call. = FALSE)


  # Check if file exists
  if(write_file){
    file_to_save <- paste0(output_dir, "/gbif_occ.", file.format)
    # Check if file exists
    if(file.exists(file_to_save) && !overwrite){
      stop("The file '", file_to_save, "' already exists. Delete the file or set 'overwrite = TRUE'")
    }}



  # Download records to temporary location
  temporary_file <- rgbif::occ_download_get(key = request_key,
                                            path = normalizePath(tempdir()),
                                            overwrite = overwrite)
  # Check columns to import
  if(select_columns){
    if(is.null(columns_to_import)){
      c_imp <- c("scientificName", "acceptedScientificName", "occurrenceID",
                 "collectionCode", "catalogNumber", "decimalLongitude",
                 "decimalLatitude", "coordinateUncertaintyInMeters",
                 "elevation", "continent", "countryCode", "stateProvince",
                 "municipality", "locality", "verbatimLocality", "year",
                 "eventDate", "recordedBy", "identifiedBy", "basisOfRecord",
                 "occurrenceRemarks", "habitat", "datasetName", "datasetKey",
                 "speciesKey")
    } else {
      c_imp <- columns_to_import
      # Check if columns exists
      absent_columns <- setdiff(columns_to_import, colnames(temporary_file))
      if(length(absent_columns) > 0)
        stop("The following columns specified in 'columns_to_import' are absent from GBIF dataset: ", paste(absent_columns, collapse = ", "))
    }
  } else {
    c_imp <- NULL
  }

  # Import occurrences
  occ <- rgbif::occ_download_import(x = temporary_file, select = c_imp, ...)

  if(write_file){
  data.table::fwrite(occ,
                     file = file_to_save)
  }
  return(occ)
}
