#' Function to perform an EM algorithm (LDA projection + 1D KDE (nonparametric, correlation-aware))
#' for a 2-component mixture where each observation \eqn{i} has a p-value \eqn{p_i}, and an auxiliary feature vector \eqn{v_i}.
#' @param pvalues Vector of gene-level p-values.
#' @param v An auxiliary feature vector.
#' @param pi0 Mixing proportion (prior probability) that an observation belongs to the null/background component \eqn{Z = 0}
#'   in the two-component mixture.
#' @param pi1 Mixing proportion (prior probability) that an observation belongs to the alternative/signal component \eqn{Z = 1}
#'   in the two-component mixture; typically \eqn{\pi_1 = 1 - \pi_0}.
#' @param alpha0 Mixing proportion (prior probability) that an observation belongs to the alternative/signal component \eqn{Z = 1}
#'   in the two-component mixture; typically \eqn{\pi_1 = 1 - \pi_0}.
#' @param alpha1 Shape parameter of the alternative/signal p-value distribution, modeled as
#'   \eqn{p \sim \mathrm{Beta}(\alpha_1, 1)} when \eqn{Z = 1}; values \eqn{< 1} concentrate mass near 0 (enrichment of small p-values).
#' @param max.it Maximum number of iterations for the EM algorithm.
#' @param verbose A logical value defining if the iteration information should be printed or not.
#' @return Returns the posterior probability of association for each gene.
#' @name model.proj
#' @keywords internal
model.proj <- function(pvalues,v,pi0,pi1, alpha0, alpha1,max.it, verbose=verbose){
  n <- dim(v)[1]
  d <- dim(v)[2]
  
  r <- EMinfer(pvalues,pi0,pi1, alpha0, alpha1, max.it)
  alpha0 <- r$alpha0
  alpha1 <- r$alpha1
  pi <- r$pi
  z <- r$post
  y <- sLDA(z,v)
  old.z <- z
  
  density0 <- seq(0,0,length.out = n)
  density1 <- seq(0,0,length.out = n)
  
  loglik.history <- c()
  i <- 0
  while(1){
    i <- i + 1
    
    density0 <- kde(y,1-z)
    density1 <- kde(y,z)
    
    tmp0 <- log(1-pi) + log(alpha0) + (alpha0-1)*log(pvalues) + log(density0)
    tmp1 <- log(pi) + log(alpha1) + (alpha1-1)*log(pvalues) + log(density1)
    z <- 1 / (1 + exp(tmp0 - tmp1))
    
    pi <- sum(z) / n
    alpha0 <- - sum(1-z) / sum((1-z) * log(pvalues))
    alpha1 <- - sum(z) / sum(z * log(pvalues))
    
    if(sum(abs(old.z - z)) < 1e-6) break;
    if(i > 1000) break;
    old.z <- z
  }
  
  result <- list(post=z, y = y, pi=pi, alpha0=alpha0, alpha1=alpha1, density0 = density0[order(y)], density1 = density1[order(y)])
  return(result)
}