#' @title Warabandi
#' @author Harvinder Singh,
#' PhD scholar, Department of Pharmacology, PGIMER, Chandigarh (160012)
#' harvindermaan4[@]gmail.com
#' ORCID https://orcid.org/0000-0002-6349-3143
#' @description
#' To generate a roster for a week (168 hrs. or 24X7) as used in warabandi
#' system. Flow time of a watercourse to an individual farmer calculate based
#' on their holding area (bigga or hectares), This flow time roster generated by
#' this function known as "warabandi" in canal irrigated agricultural areas
#' (Punjab, Rajasthan, Haryana and Some areas of Pakistan)
#' @details
#' A regulated irrigation system, from source (reservoir or river) down to gate
#'  of farm known as 'nakka' known as warabandi. Water from reservoir or source
#'  is carried out by main canal which supplies sub-canal or sub-distributaries.
#'  Sub-distributaries run with full supply to secondary-sub-distributaries with
#'  rotations. These secondary-sub-distributaries supplies to watercourse
#'  through
#'  outlets. For water supply to different farms or fields situated along the
#'  web of watercourse by a time roster of week i.e., "Warabandi". Watercourse
#'  runs at full supply when secondary-sub-distributaries are in flow. This is
#'  3° distributary system. The main objective of warabandi is to attain maximum
#'   efficiency of water use by implementing water scarcity on every user.
#'   Warabandi justifies equality and safe guards to every farmer even in the
#'    case of farm is located at the end of the watercourse. The roster of turn
#'    in the first we have to define various terminology used in warabandi;
#'
#' Bharai:
#'
#' When water start to fill-up the empty watercourse, time spent to fill
#' the empty watercourse (length of the head to the farm gate i.e., Nakka) or we
#' can say filling time or compensation to the farmer, known as 'bharai'.
#'
#' Jharai:
#'
#' When tail or end of the watercourse farmer has his turn and head or at
#' the start of watercourse farmer diverted his shared water then rest of the
#' flowed water will goes to the tail end farmer as compensation, i.e., empty
#' time.
#'
#' Rakva:
#'
#' Culturable area owned by farmer or Area subjected to irrigation.
#' Flow Time per unit area = It is a calculatable time by using following
#' formula;
#'
#' "168 hours - (Total bharai - Total jharai) / Total culturable command area".
#'
#' Flow time for farmer<- It is a calculatable time by following formula;
#'
#'    (Flow time per unit area X Area owned by farmer) + (bharai for his field's
#'     gate location - jharai for his field's gate location)
#'
#' Roster of turn
#'
#' Basically, warabandi or roster of turn starts from Monday, 6:00 PM to next
#' Monday, 6:00 PM throughout all week days. Time and date calculation can be
#' done by “lubridate” R package, but it’s not able to perform divide and
#' multiplication tasks with numeric. In our knowledge there is no software or
#' package to generate directly warabandi based on data available with “patwari”.
#' Data required to generate the warabandi can be found from related irrigation
#' department officer known as "Patwari".
#' @note Data structure:
#' There should be at least 4 column containing header with;
#' 1. Names of Farmer
#'
#' 2. Rakva: There can be multiple columns for this section but be sure header
#'   should be specified as (x.1, x.2, x.3.....x.N)
#'
#' 3. Bharai: This column should have data of compensation in the format of H:M
#'    (1:10). If there is no compensation to give a farmer then put it as '0:0'.
#'
#' 4. Jharai: This column should have data of Jharai in the same format as
#'    described in case of bharai. If there is no jharai to deduct from a farmer
#'     flow time then put it as '0:0'.
#'
#' @note
#' Input data should be in the ".txt" or ".csv" format. Which can be generated
#' in in any data entry software like excel, libra etc.
#' That's enough to calculate warabandi and it will generate full a week roaster
#'  as Output.csv. Output file will be saved in working directory
#'
#'  Required external packages:
#'  a) “lubridate”
#'  b) “readtext”
#'  d) “flextable”
#'  Or
#'  It can be said:
#'  Imports:
#'  lubridate
#'  readtext
#'  flextable
#'
#'@keywords warabandi
#'@keywords roster  for weekdays
#'@kewords Flow time
#'
#'@references
#'1. G. Asawa.Irrigation and water resources engineering. New Age International,
#'   2006
#'
#'2. Bandaragoda DJ(1995) <https://publications.iwmi.org/pdf/H_17571i.pdf>
#'
#'3. V.  Narain.   Warabandi  as  a  sociotechnical  system  for  canal  water
#'  allocation:  opportunities  and challenges for reform.Water Policy,
#'  10(4):409â422, 2008
#'
#'4. Ajmera S, Shrivastava RK. Water distribution schedule under warabandi
#'   system considering seepage losses for an irrigation project: A case study.
#'   Intl. J. Innov. Eng. Tech. 2013;2(4):178-87.
#'
#' @concept wararabandi
#' @concept Roster
#' @concept weekdays
#' @concept Flow time per unit area
#' @return A list of different objects i.e. Flow Time per Unit Area, Total rakva,
#' Total bharai, Total jharai, Full week Hours and Final report
#' To generate final report saved output file has to be supplied into
#' My_file.Rmd by editing it as per your need i.e. format of output document or
#'  headers and footers for the output table report.
#' @param file = Contains data for calculation
#' @param output = TRUE or FALSE to write output file
#' @param nof = "My_report" specify output file name
#' @import utils
#' @export
warabandi<-function(file, output= c(TRUE, FALSE), nof){
  #library(rJava)
  if(is.data.frame(file)){in.dt <- file}
  else if( grepl(".txt", file )){in.dt <- readtext::readtext(file = file,
                                                        header = T)}
  else if(grepl(".csv", file)){in.dt <- utils::read.csv(file = file,
                                                        header = T)}
  Nakka <- in.dt[, grep(pattern = '^Nk.*', names(in.dt))]
  P1 <- in.dt[, grep(pattern = '^x.*', names(in.dt))]
  P1[is.na(P1)] <- 0
  P2 <- in.dt[, grep(pattern = '^M.*', names(in.dt))]
  P2[is.na(P2)] <- 0
  P3 <- t(P2)
  P5 <- t(P1)
  D_rkva <- apply(P5, 2, paste, collapse=", ")
  MuNo <- apply(P3, 2, paste, collapse=", ")
  Drkva <- paste(D_rkva, "\n", "MNo", MuNo)


  rakva_only <- P1
  bharai <- as.POSIXlt(strptime(in.dt$Bharai, "%H:%M"))
  bharai[is.na.POSIXlt(bharai)] <- as.POSIXlt(strptime("0:0", "%H:%M"))
  jharai <- as.POSIXlt(strptime(in.dt$Katai, "%H:%M"))
  jharai[is.na.POSIXlt(jharai)] <- as.POSIXlt(strptime("0:0", "%H:%M"))


  Individual_rakva_sum <- rowSums(rakva_only)
  Total_rakva <- sum(Individual_rakva_sum)
  total_bharai_microseconds <- (sum(bharai$hour) * 3600000000) +
    (sum(bharai$min) * 60000000)
  total_jharai_microseconds <- (sum(jharai$hour) * 3600000000) +
    (sum(jharai$min) * 60000000)
  total_microsecond_in_a_week <- 604800000000
  Total_bharai <- as.character(lubridate::seconds_to_period(
    total_bharai_microseconds / 1000000))
  Total_jharai <- as.character(lubridate::seconds_to_period(
    total_jharai_microseconds / 1000000))

  FlowTime_per_Unit_Area_microseconds <- (((total_microsecond_in_a_week -
                    total_bharai_microseconds) +
                    total_jharai_microseconds) / (Total_rakva))

  tpb <- lubridate::seconds_to_period(FlowTime_per_Unit_Area_microseconds
                                      / 1000000)

  FlowTime_per_Unit_Area <- lubridate::minute(tpb)

  total_FlowTime_microseconds <- (Individual_rakva_sum *
                                    FlowTime_per_Unit_Area_microseconds)

  jharai_in_microseconds <-(((jharai$hour) * 3600000000) + (jharai$min)
                            * 60000000)


  bharai_in_microseconds <- (((bharai$hour) * 3600000000) + (bharai$min) *
                               60000000)

  actual_FlowTime_in_microseconds <- ((total_FlowTime_microseconds +
                                         bharai_in_microseconds) -
                                        jharai_in_microseconds)

  Full_week_H <- as.character(lubridate::seconds_to_period(sum(
    actual_FlowTime_in_microseconds) / 1000000))


  actual_FlowTime_in_minutes <- lubridate::seconds_to_period(
    actual_FlowTime_in_microseconds / 1000000)

  actual_FlowTime_H <- paste(lubridate::hour(actual_FlowTime_in_minutes),
                             lubridate::minute(actual_FlowTime_in_minutes),
                             sep = ":")

  actual_FlowTime_H

  Cum_sum_list <- cumsum(actual_FlowTime_in_microseconds)

  SEconds <- (Cum_sum_list/1000000)

  Final_time_table <- lubridate::ymd_hm("2021-05-24 18:00") + SEconds

  round.POSIXt(Final_time_table, units = "mins")

  exp7 <- (Cum_sum_list) - (actual_FlowTime_in_microseconds)

  Seconds2 <- (exp7/1000000)

  Final_time_table2 <- lubridate::ymd_hm("2021-05-24 18:00") + Seconds2

  round.POSIXt(Final_time_table2, units = "mins")

  time_breakage_1_day2 <- paste(lubridate::hour(Final_time_table2),
                                lubridate::minute(Final_time_table2), sep=":")

  Pani_lene_ka_time <- paste(weekdays.Date(Final_time_table2),
                             time_breakage_1_day2, sep = ', ')

  time_breakage_1_day <- paste(lubridate::hour(Final_time_table),
                               lubridate::minute(Final_time_table), sep=":")

  Start_from_Monday_evening_6_pm <- paste(weekdays.Date(Final_time_table),
                                          time_breakage_1_day, sep = ', ')

  Time_schedule <- paste(Pani_lene_ka_time, Start_from_Monday_evening_6_pm,
                         sep = "-")


  RakvaSum <- Individual_rakva_sum; FTPUA <- FlowTime_per_Unit_Area;
  TotalTurn <- actual_FlowTime_H; Name <- in.dt$Name_of_Farmer;
  Bharai <- in.dt$Bharai; Jharai <- in.dt$Katai



  CompleteTable4 <- data.frame(Name, Nakka, Drkva, RakvaSum, TotalTurn,
                               Bharai, Jharai, Time_schedule)






  ########## Witting the output

  if(output == TRUE){
    outputname<-as.character(nof)
    utils::write.csv(CompleteTable4, file = paste(outputname))
  } else {
    print(CompleteTable4)
  }

  final_list_Report<-list(FlowTime_per_Unit_Area, Total_rakva,
                          Total_bharai, Total_jharai, Full_week_H,
                          CompleteTable4)
  final_list_Report
}
