% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ate_specs.R
\name{ate_from_bv}
\alias{ate_from_bv}
\title{Calculate Allowable Total Error from Biological Variation}
\usage{
ate_from_bv(
  cvi,
  cvg = NULL,
  level = c("desirable", "optimal", "minimum"),
  k = 1.65
)
}
\arguments{
\item{cvi}{Numeric. Within-subject (intra-individual) biological variation
coefficient of variation, expressed as a percentage.}

\item{cvg}{Numeric. Between-subject (inter-individual) biological variation
coefficient of variation, expressed as a percentage. If \code{NULL} (default),
only imprecision specifications are calculated.}

\item{level}{Character. Performance level: \code{"desirable"} (default),
\code{"optimal"}, or \code{"minimum"}. See Details.}

\item{k}{Numeric. Coverage factor for total allowable error calculation
(default: 1.65 for ~95\% coverage assuming normal distribution).}
}
\value{
An object of class \code{c("ate_specs", "valytics_ate", "valytics_result")},
which is a list containing:

\describe{
\item{specifications}{List with calculated specifications:
\itemize{
\item \code{allowable_cv}: Allowable analytical imprecision (CV_A)
\item \code{allowable_bias}: Allowable analytical bias (NULL if cvg not provided)
\item \code{tea}: Total allowable error (NULL if cvg not provided)
}
}
\item{input}{List with input parameters:
\itemize{
\item \code{cvi}: Within-subject CV
\item \code{cvg}: Between-subject CV (or NULL)
\item \code{level}: Performance level used
\item \code{k}: Coverage factor used
}
}
\item{multipliers}{List with level-specific multipliers used:
\itemize{
\item \code{imprecision}: Multiplier for CV_I
\item \code{bias}: Multiplier for sqrt(CV_I^2 + CV_G^2)
}
}
}
}
\description{
Calculates analytical performance specifications (allowable imprecision,
allowable bias, and total allowable error) based on biological variation
data using the hierarchical model from Fraser & Petersen (1993).
}
\details{
The biological variation model for analytical performance specifications
was developed by Fraser, Petersen, and colleagues. The model derives
allowable analytical error from the inherent biological variability of
the measurand.

\strong{Formulas (Desirable level):}

\deqn{CV_A \leq 0.50 \times CV_I}{CV_A <= 0.50 * CV_I}

\deqn{Bias \leq 0.25 \times \sqrt{CV_I^2 + CV_G^2}}{Bias <= 0.25 * sqrt(CV_I^2 + CV_G^2)}

\deqn{TEa \leq k \times CV_A + Bias}{TEa <= k * CV_A + Bias}

Where:
\itemize{
\item CV_I = within-subject biological variation
\item CV_G = between-subject biological variation
\item CV_A = allowable analytical imprecision
\item k = coverage factor (default 1.65)
}

\strong{Performance Levels:}

Three hierarchical performance levels are defined:
\itemize{
\item \strong{Optimal}: Most stringent; multipliers are 0.25x desirable
(i.e., 0.125 for CV, 0.0625 for bias)
\item \strong{Desirable}: Standard target; multipliers are 0.50 for CV,
0.25 for bias
\item \strong{Minimum}: Least stringent; multipliers are 1.5x desirable
(i.e., 0.75 for CV, 0.375 for bias)
}
}
\section{Data Sources}{

Biological variation data (CV_I and CV_G) should be obtained from
authoritative sources. The recommended current source is the
\strong{EFLM Biological Variation Database}: \url{https://biologicalvariation.eu/}

This database provides rigorously reviewed BV estimates derived from
published studies meeting defined quality specifications.
}

\examples{
# Glucose: CV_I = 5.6\%, CV_G = 7.5\% (example values)
ate <- ate_from_bv(cvi = 5.6, cvg = 7.5)
ate

# Optimal performance level (more stringent)
ate_optimal <- ate_from_bv(cvi = 5.6, cvg = 7.5, level = "optimal")
ate_optimal

# Minimum acceptable performance
ate_min <- ate_from_bv(cvi = 5.6, cvg = 7.5, level = "minimum")
ate_min

# When only within-subject CV is known (bias goal not calculable)
ate_cv_only <- ate_from_bv(cvi = 5.6)
ate_cv_only

# Custom coverage factor (e.g., 2.0 for ~97.5\% coverage)
ate_custom <- ate_from_bv(cvi = 5.6, cvg = 7.5, k = 2.0)

# Access individual specifications
ate$specifications$allowable_cv
ate$specifications$allowable_bias
ate$specifications$tea

}
\references{
Fraser CG, Petersen PH (1993). Desirable standards for laboratory tests
if they are to fulfill medical needs. \emph{Clinical Chemistry},
39(7):1447-1453.

Ricos C, Alvarez V, Cava F, et al. (1999). Current databases on biological
variation: pros, cons and progress. \emph{Scandinavian Journal of Clinical
and Laboratory Investigation}, 59(7):491-500.

Aarsand AK, Fernandez-Calle P, Webster C, et al. (2020). The EFLM
Biological Variation Database. \url{https://biologicalvariation.eu/}

Westgard JO (2008). \emph{Basic Method Validation} (3rd ed.).
Westgard QC, Inc.
}
\seealso{
\code{\link[=sigma_metric]{sigma_metric()}} for calculating Six Sigma metrics,
\code{\link[=ate_assessment]{ate_assessment()}} for comparing observed performance to specifications
}
