% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cirWrappers.r
\name{udplot}
\alias{udplot}
\title{Visualizing the time series of an up-and-down experiment}
\usage{
udplot(
  x,
  y,
  cohort = NULL,
  shape = "circle",
  connect = TRUE,
  symbcol = 1,
  doselabels = NULL,
  allow1extra = FALSE,
  xtitle = "Observation Order",
  ytitle = "Dose / Stimulus",
  ...
)
}
\arguments{
\item{x}{numeric vector: sequence of administered doses, treatments, stimuli, etc.}

\item{y}{numeric vector: sequence of observed responses. Must be same length as \code{x}, and must be coded \code{TRUE/FALSE} or 0/1.}

\item{cohort}{for a group/cohort UD design, the cohort/group size (a single number). In case of variable cohort size, this can be a vector the same length as \verb{x, y}, with each observation's cohort assignment.}

\item{shape}{the plotting shape (DRtrace only): \code{'circle'} (default), \code{'square'}, or \code{'triangle'}.}

\item{connect}{logical: whether to connect the symbols (generic plotting type \code{'b'}). Default \code{TRUE} for \code{udplot()} and \code{FALSE} for \code{drplot()}.}

\item{symbcol}{The color of the main plotting symbols and connecting lines. Default 1 (the current palette's first color). Note: if you change the color and inadvertently use \code{col} instead, there might be an error message.}

\item{doselabels}{(\code{DRtrace} only) Dose values to be plotted along the y-axis. If \code{NULL} (default), those will be the doses in the dataset (i.e., \code{sort(unique(x))}).}

\item{allow1extra}{logical: allow \code{length(x)} to be either equal or 1 greater than \code{length(y)}? (default \code{FALSE}) The \emph{"n+1"} dose-allocation, determined from the last allocations and responses, might be tagged onto \code{x}. If this point is provided and \code{allow1extra=TRUE}, \code{udplot()} will show it as a grey diamond; the other functions will ignore it.}

\item{xtitle, ytitle}{x-axis and y-axis titles. Some reasonable defaults are provided, to avoid an annoying error message.}

\item{...}{Other arguments passed on to \code{\link[graphics]{plot}} (e.g., \code{main} for the main title).}
}
\value{
Returns invisibly after plotting. If you would like to save the plot to a file, embed the plotting code in a standard R graphics export code sequence, (e.g., \code{pdf(...)} before the plotting function, and \code{dev.off()} after it).
}
\description{
Plotting function for the "trace" (time series) of an up-and-down experiment, showing the observation order on the x-axis, and the dose \emph{(treatment, stimulus, etc.)} strength on the y-axis. Uses utilities from the \code{cir} package.
}
\details{
This simple and handy visualization approach was presented already by Dixon and Mood (1948).
\itemize{
\item It conveys directly the meaning of \emph{"up-and-down"}, because the administered dose/stimulus strength is on the y-axis, whereas observation order is on the x-axis.
\item Filled symbols stand for positive responses and open symbols for negative.
\item The design's transition rules can be usually inferred directly from the plot.
}

\code{udplot()} is a convenience wrapper to \code{cir::plot.DRtrace}. This is a base-R plot, so you can use additional options, including preceding the plot command with \code{\link[graphics]{par}} statements, or following up with \code{\link[graphics]{legend}}. When wishing to save to a file, I recommend utilities such as \code{png()} or \code{pdf()}.
}
\examples{

#'  **An up-and-down experiment that has generated some controversy**
#'  
#' Van Elstraete, AC et al. The Median Effective Dose of Preemptive Gabapentin 
#'      on Postoperative Morphine Consumption After Posterior Lumbar Spinal Fusion. 
#'      *Anesthesia & Analgesia* 2008, 106: 305-308.

# It was a classical median-finding up-and-down study.

doses = c(4:7, 6:13, 12:19, 18:21, 20, 19:23, 22, 21:23, 22:19, 20:23, 
          22:24, 23, 22, 23, 22:25, 24:22, rep(23:24,2), 23, 22)
# With U&D, responses (except the last one) can be read off the doses:
responses = c( (1 - sign(diff(doses)))/2, 0 )


#' ### Plots plots plots!

# Saving current settings as now required by the CRAN powers-that-be :0
op <- par(no.readonly = TRUE)

layout(t(1:2), widths=3:2)
par(mar=c(4,4,4,1), mgp=c(2.5,0.8,0), cex.axis = 0.7, las = 1)

#' The experimental trajectory / time-series / "trace" (pick your favorite name!)
#' Note the changed argument names for x and y axis titles
udplot(doses, responses, main='', 
        xtitle = "Patient Number", ytitle = 'Gabapentin (mg/kg)') 
#' Compare with the article's Figure 1; the line below makes it look more similar
udplot(doses, responses, shape='square', connect=TRUE)

# The dose-response plot, rarely encountered in U&D articles. 
# We can also add the CIR estimate right there:
drplot(doses, responses, main=' Dose-Response', percents = TRUE,
       addest = TRUE, target = 0.5, addcurve = TRUE,
       xtitle = 'Gabapentin (mg/kg)', ytitle = "Percent Effective")

#' ### Estimates

#' Let us actually see the numbers of those Centered-Isotonic-Regression (CIR) estimates!
#' Note that our default confidence-interval is 90\%. Change it via the 'conf' argument.

udest(doses, responses, target = 0.5)
#' Compare with the article: 21.7 mg/kg (95\% CI 19.9–23.5). 
#' They cite a little-known 1991 article by Dixon as the method source.
#' However, in their author rejoinder they claim to have used the Dixon-Mood estimate.
#' 
#' ## Toy example of plotting a group UD dataset
#' 
#' Also showing off some udplot() options
#' 
#' Not an actual experiment (made-up data)
#' The design is purportedly GUD (3,0,1), targeting the 20th percentile
#' 

gsize = 3
x = rep(c(1:3, 2:4), each = gsize)
y = c(rep(0, 8), 1, rep(0,7), 1, 1)

udplot(x=x, y=y, cohort=gsize, connect=FALSE, shape='triangle')

par(op) # Back to business as usual ;)
}
\references{
\itemize{
\item Dixon WJ, Mood AM. A method for obtaining and analyzing sensitivity data. \emph{J Am Stat Assoc.} 1948;43:109-126.
\item Oron AP, Souter MJ, Flournoy N. Understanding Research Methods: Up-and-down Designs for Dose-finding. \emph{Anesthesiology} 2022; 137:137–50.
}
}
\seealso{
\itemize{
\item \code{\link[cir]{plot.DRtrace}}, \code{cir} package.
\item \code{\link{drplot}} for the up-and-down dose-response and estimate plotting.
\item \code{cir} package vignette.
}
}
\author{
Assaf P. Oron \code{<assaf.oron.at.gmail.com>}
}
