library(testthat)
library(dplyr)

# Cox model for testing
cox_fit <- coxph(Surv(AVAL, EVENT) ~ TRT01P, ties = "exact", data = codebreak200)
# tipse object
results <- tipping_point_model_free(
  dat = codebreak200,
  reason = "Early dropout",
  impute = "docetaxel",
  J = 2,
  tipping_range = seq(10, 80, by = 10),
  cox_fit = cox_fit,
  method = "random sampling",
  seed = 123
)

test_that("summary returns correct structure for random sampling", {
  res <- summary(results)

  expect_s3_class(res, "data.frame")
  expect_true(all(c("HR", "CONFINT", "METHOD", "ARMIMP", "TIPPT", "TIPUNIT", "DESC") %in% names(res)))
  expect_equal(res$METHOD, "random sampling")
  expect_true(grepl("percentile", res$TIPUNIT))
})


test_that("summary returns correct structure for deterministic sampling", {
  # tipse object
  results2 <- tipping_point_model_free(
    dat = codebreak200,
    reason = "Early dropout",
    impute = "docetaxel",
    J = 2,
    tipping_range = seq(1, 21, by = 5),
    cox_fit = cox_fit,
    method = "deterministic sampling",
    seed = 123
  )
  res <- summary(results2)

  expect_equal(res$METHOD, "deterministic sampling")
  expect_true(grepl("number of subjects", res$TIPUNIT))
})


test_that("summary handles hazard inflation and hazard deflation", {
  # tipse object
  results3 <- tipping_point_model_based(
    dat = codebreak200,
    reason = "Early dropout",
    impute = "sotorasib",
    J = 2,
    tipping_range = seq(1, 2, by = 0.05),
    cox_fit = cox_fit,
    seed = 123
  )
  res_infl <- summary(results3)
  expect_true(grepl("hazard inflation", res_infl$TIPUNIT))

  # tipse object
  results4 <- tipping_point_model_based(
    dat = codebreak200,
    reason = "Early dropout",
    impute = "docetaxel",
    J = 2,
    tipping_range = seq(0.1, 1, by = 0.05),
    cox_fit = cox_fit,
    seed = 123
  )

  res_defl <- summary(results4)
  expect_true(grepl("hazard deflation", res_defl$TIPUNIT))
})

test_that("summary handles case with no tipping point detected", {
  # tipse object
  expect_warning(results5 <- tipping_point_model_free(
    dat = codebreak200,
    reason = "Early dropout",
    impute = "sotorasib",
    J = 2,
    tipping_range = c(20, 40),
    cox_fit = cox_fit,
    method = "random sampling",
    seed = 123
  ), "Tipping point not found, please check 'tipping_range'.")
  res <- summary(results5)

  expect_true(is.na(res$TIPPT))
  expect_true(grepl("No tipping point detected", res$DESC))
})

test_that("summary formats CI and interpretation correctly", {
  res <- summary(results)

  expect_true(grepl("\\(", res$CONFINT)) # CI formatted with parentheses
  expect_true(grepl("Tipping point", res$DESC))
})
