#' @importFrom sf st_as_sf st_sf 
#' @importFrom dplyr "%>%"
#' @importFrom leaflet leaflet highlightOptions addControl addMarkers addPolygons addTiles fitBounds addCircles labelOptions renderLeaflet leafletOutput clearGroup addLayersControl layersControlOptions hideGroup setView addProviderTiles
# #' @importFrom leaflet.extras addDrawToolbar editToolbarOptions selectedPathOptions
#' @importFrom leaftime addTimeline timelineOptions styleOptions
#' @importFrom terra xmin xmax ymin ymax 
#' @importFrom htmlwidgets saveWidget
#' @importFrom geojsonio geojson_json
#' @importFrom htmltools tags HTML 
#' 
#' 
#' 
NULL
#' 
#' 
#' @title exploring
#' 
#' @description explore identification of islands and naming 
#' 
#' @param labs labeling dataset
#' @param aslv reconstructed dataset 
#' @param path NULL; path where to store the output html
#'
#' @return html file
#' 
#'
#' @noRd
#' @keywords internal
#'
#'
exploring <- function(labs, aslv, simplify=TRUE, path=NULL){
  
  isl_style <- reclass_names(labs$name)
  asl_style <- reclass_names(aslv$recname)
  
  col_labs <-data.frame(class=c('^UNKNOWN$','^UNNAMED$','^I$','^N$'),
                             col=c('darkorange4','darkgoldenrod','steelblue','steelblue'),
                             col_locs=c('red','red','red','red'),
                             col_landsat=c('darkgrey','lightgrey',NA,NA),
                             stringsAsFactors = FALSE)
  asl_style$col <- '#80cdc1' # lightgreen
  asl_style$colbor <- '#01665e' # darkgreen
  asl_style$col_locs <- 'black' # black
  isl_style$col <- '#dfc27d' # lightbrown
  isl_style$colbor <- '#8c510a' # darkbrown
  for(i in 1:nrow(col_labs)){
    asl_style[grepl(col_labs[i,'class'], asl_style$class),'col'] <- col_labs[i,'col']
    asl_style[grepl(col_labs[i,'class'], asl_style$class),'colbor'] <- col_labs[i,'col']
    asl_style[grepl(col_labs[i,'class'], asl_style$class),'col_locs'] <- col_labs[i,'col_locs']
    isl_style[grepl(col_labs[i,'class'], isl_style$class),'col'] <- col_labs[i,'col_landsat']
    isl_style[grepl(col_labs[i,'class'], isl_style$class),'colbor'] <- col_labs[i,'col_landsat']
  }
  asl_polys <- suppressWarnings(sf::st_as_sf(aslv))
  isl_polys <- suppressWarnings(sf::st_as_sf(labs))
  asl_peaks <- suppressWarnings(sf::st_as_sf(as.data.frame(aslv),coords=c('x','y'),crs=crs(aslv)))
  isl_peaks <- suppressWarnings(sf::st_as_sf(as.data.frame(labs),coords=c('refx','refy'),crs=crs(labs)))

  if(simplify){
  asl_polys  <- suppressWarnings(sf::st_simplify(asl_polys, preserveTopology = TRUE, dTolerance = 0.001))
  isl_polys  <- suppressWarnings(sf::st_simplify(isl_polys, preserveTopology = TRUE, dTolerance = 0.001))
  }
  
  tag.map.title <- htmltools::tags$style(htmltools::HTML(".leaflet-control.map-title {
                                            position: absolute;
                                            left: 100%;
                                            text-align: left;
                                            padding-left: 10px;
                                            width: 70px;
                                            white-space: nowrap;
                                            bottom:100;
                                            color:rgb(47,79,79);
                                            padding-right: 10px;
                                            font-weight: bold;
                                            font-size: 9.5px;
                                          }
                                        "))
  
  title <- htmltools::tags$div(tag.map.title, htmltools::HTML('&copy; De Groeve, J., Rijsdijk, K.F., Rentier, E.S., Flantua, S.G.A., Norder, S.J. (2025)'))
  
  peakname <- 'locations' 
  leaf <- leaflet::leaflet() %>%
    leaflet::addTiles(urlTemplate="http://server.arcgisonline.com/ArcGIS/rest/services/World_Imagery/MapServer/tile/{z}/{y}/{x}",group="esri.sat") %>%
    # OSM in local language
    leaflet::addTiles(group = "OSM") %>%
    leaflet::addControl(title,  className='map-title') %>% 
    leaflet::addProviderTiles(provider = leaflet::providers$CartoDB.Voyager,group = "carto") %>%
    leaflet::addTiles(urlTemplate='https://server.arcgisonline.com/ArcGIS/rest/services/World_Topo_Map/MapServer/tile/{z}/{y}/{x}',group = "esri.topo") %>%
    leaflet::addTiles(group = 'transparent', options=leaflet::tileOptions(opacity=0)) %>%
    leaflet::addPolygons(data = asl_polys,
                         fillColor =asl_style$col,
                         color=asl_style$colbor,
                         opacity = 0.8,
                         fillOpacity = 0,weight=1,
                         highlightOptions = leaflet::highlightOptions(color = 'white',
                                                                      weight = 1,
                                                                      bringToFront = TRUE),
                         group='reconstruction') %>%
    leaflet::addCircles(data=asl_peaks,
                        popup = asl_style$name,
                        color= asl_style$col_locs,
                        label =  ~as.character(recname),
                        labelOptions = leaflet::labelOptions(noHide = TRUE, textOnly = TRUE), 
                        group=peakname) %>%
    # Layers control
    leaflet::addLayersControl(
      baseGroups = c("carto","esri.sat", "OSM", "esri.topo",'transparent'),
      overlayGroups = c("reconstruction","labs",peakname), #
      options = leaflet::layersControlOptions(collapsed = FALSE)) %>% 
    leaflet::hideGroup(peakname)
  
    if (inherits(isl_polys$geometry, "sfc_POLYGON") | inherits(isl_polys$geometry, "sfc_MULTIPOLYGON") | grepl('POLYGON',sf::st_geometry_type(isl_polys)[1])) {
      leaf <- leaf %>% leaflet::addPolygons(data = sf::st_transform(isl_polys,sf::st_crs(asl_polys)),
                           fillColor =isl_style$col,
                           color=isl_style$colbor,
                           opacity = 0.8,
                           fillOpacity = 0,
                           weight=1,
                           highlightOptions = leaflet::highlightOptions(color = 'white',
                                                                        weight = 1,
                                                                        bringToFront = TRUE),
                           group='labs')
    } else {
      leaf <- leaf %>% leaflet::addCircles(data = sf::st_transform(isl_polys,sf::st_crs(asl_polys)),
                          fillColor =isl_style$col,
                          color=isl_style$colbor,
                          opacity = 0.8,
                          fillOpacity = 0,
                          weight=1,
                          highlightOptions = leaflet::highlightOptions(color = 'white',
                                                                       weight = 1,
                                                                       bringToFront = TRUE),
                          group='labs')      
    }
  
  #message('test')
  #message(leaf)
  #message('test')
  
  return(leaf)
}


#' timelapse
#' @title timelapse 
#' 
#' @description generate a timelapse for a reconstructed island dataset
#' 
#' @param paleostack islandstack generated by make_island_stack
#' @param simplify boolean (TRUE/FALSE) if simplify is TRUE than a smoothed polygon is generated for visualisation purposes
#' @param overlay How many periods overlap in the timelapse
#' @param speed numeric, speed of the animation, 1 to 10. 
#'
#' @return html file
#'
#' @noRd
#' @keywords internal
#'
#' 
#'
timelapse <- function(paleostack, simplify=TRUE, overlay=2, speed=1){
  
  # robustify simplify
  if (is.null(simplify)) simplify <- TRUE
  simplify <- isTRUE(simplify)
  
  # speed (1 to 10)
  if (isTRUE(speed)) speed <- 1
  if (is.null(speed) || isFALSE(speed)) speed <- 1
  speed <- suppressWarnings(as.numeric(speed))
  if (is.na(speed)) speed <- 1
  speed <- max(1, min(10, speed))
  
  # spatial objects 
  polys <- lapply(paleostack, function(x) sf::st_as_sf(x))
  if (simplify) {
    polys <- lapply(polys, function(x)
      suppressWarnings(sf::st_simplify(x, preserveTopology = TRUE, dTolerance = 0.001))
    )
  }
  
  # Parse labels from fixed-width names: BP9999999 / AP0000500
  nm  <- names(paleostack)
  era <- substr(nm, 1, 2)  # "BP" or "AP"
  yr  <- suppressWarnings(as.integer(substr(nm, 3, nchar(nm))))
  yr[is.na(yr)] <- 0
  
  # display labels (whole numbers; AP shown as AD)
  period_labels <- ifelse(era == "BP",
                          paste0(yr, " BP"),
                          paste0(yr, " AD"))
  
  # numeric period (BP negative) only used for scaling
  extract_period <- yr
  extract_period[era == "BP"] <- -extract_period[era == "BP"]
  
  # scaling 
  extract_period_scaled <- ((extract_period - min(extract_period)) /
                              (max(extract_period) - min(extract_period)) *
                              (length(extract_period) - 1) + 1) * (overlay + 1)
  
  polysdf <- lapply(polys, function(x) sf::st_sf(x))
  for(i in seq_along(polysdf)){
    polysdf[[i]]$start <- extract_period_scaled[i]
  }
  power_poly <- do.call(rbind.data.frame, polysdf)
  power_poly$end <- power_poly$start + overlay
  
  power_geo_poly <- geojsonio::geojson_json(power_poly, lat='y', lon='x', pretty=TRUE)
  
  # slider label mapping: match slider value to nearest start 
  starts_json <- jsonlite::toJSON(as.numeric(extract_period_scaled), auto_unbox = TRUE)
  labels_json <- jsonlite::toJSON(as.character(period_labels), auto_unbox = TRUE)
  
  formatOutputFun <- htmlwidgets::JS(sprintf(
    "function(v){
       var starts = %s;
       var labels = %s;
       var best = 0;
       var bestd = Infinity;
       for (var i=0; i<starts.length; i++){
         var d = Math.abs(v - starts[i]);
         if (d < bestd){ bestd = d; best = i; }
       }
       return labels[best];
     }",
    starts_json, labels_json
  ))
  
  # Animation speed (natural/exponential) 
  n_layers <- length(polys)
  
  # range of ms-per-layer:
  # speed=1  -> 300 ms/layer (slowest)
  # speed=10 -> 40  ms/layer (fastest)
  slow_ms <- 300
  fast_ms <- 40
  
  # Exponential interpolation between slow_ms and fast_ms
  # speed in 1 to 10 
  t <- (speed - 1) / 9
  ms_per_layer <- round(slow_ms * (fast_ms / slow_ms)^t)
  
  duration_ms <- as.integer(max(200, n_layers * ms_per_layer))
  
  slideropts <- leaftime::sliderOptions(
    formatOutput = formatOutputFun,
    enableKeyboardControls = TRUE,
    steps = n_layers,        # one step per layer
    duration = duration_ms,  # total playback length
    showTicks = FALSE        # no ticks
  )
  
  # title/logo 
  tag.map.title <- htmltools::tags$style(htmltools::HTML(".leaflet-control.map-title {
    position: absolute; left: 100%; text-align: left;
    padding-left: 10px; width: 70px; white-space: nowrap;
    bottom:100; color:rgb(47,79,79);
    padding-right: 10px; font-weight: bold; font-size: 9.5px;
  }"))
  title <- htmltools::tags$div(tag.map.title,
                               htmltools::HTML('&copy; De Groeve, J., Rijsdijk, K.F., Rentier, E.S., Flantua, S.G.A., Norder, S.J. (2025)'))
  ibeduva <- "<img src='https://palaeolim.files.wordpress.com/2018/02/ibed-2018-uva-logo.jpg?w=580' style='width:83px;height:83px;'>"
  
  draw <- " leaflet::addScaleBar(position='bottomright') %>% leaflet::addMeasure(position = 'topleft',primaryLengthUnit='meters',secondaryLengthUnit='kilometers',primaryAreaUnit='sqmeters',secondaryAreaUnit='hectares') %>% "
  
  time_leaf <- " leaftime::addTimeline(
      data = power_geo_poly,
      timelineOpts = leaftime::timelineOptions(
        styleOptions = leaftime::styleOptions(
          radius = 2, color =  '#01665e', stroke = TRUE,
          fill = '#80cdc1', fillColor = '#80cdc1', fillOpacity = NULL
        )
      ),
      sliderOpts = slideropts
    ) "
  
  c <- paste0(
    "leaf <- leaflet::leaflet() %>% ", draw, "
     leaflet::addControl(html=ibeduva, position='topright') %>%
     leaflet::addControl(title, className='map-title') %>%
     leaflet::addTiles(urlTemplate='http://server.arcgisonline.com/ArcGIS/rest/services/World_Imagery/MapServer/tile/{z}/{y}/{x}', group='esri.sat') %>%
     leaflet::addTiles(group='OSM') %>%
     leaflet::addProviderTiles(provider=leaflet::providers$CartoDB.Voyager, group='carto') %>%
     leaflet::addTiles(urlTemplate='https://server.arcgisonline.com/ArcGIS/rest/services/World_Topo_Map/MapServer/tile/{z}/{y}/{x}', group='esri.topo') %>%
     leaflet::addTiles(group='transparent', options=leaflet::tileOptions(opacity=0)) %>%
     leaflet::fitBounds(lng1=xmin(paleostack[[1]]), lat1=ymin(paleostack[[1]]),
                        lng2=xmax(paleostack[[1]]), lat2=ymax(paleostack[[1]])) %>% ",
    paste0(
      "leaflet::addPolygons(data=polys[[", base::seq_along(polys), "]],
        color='#01665e', stroke=TRUE, weight=0.4,
        fill='#80cdc1', fillColor='#80cdc1', fillOpacity=NULL,
        label=~as.character(recname),
        labelOptions=leaflet::labelOptions(noHide=FALSE, textOnly=TRUE),
        group='", extract_period, "') %>% ",
      collapse=' '
    ),
    "leaflet::addLayersControl(
        baseGroups=c('carto','esri.sat','OSM','esri.topo','transparent'),
        overlayGroups=c(", paste0("'", extract_period, "'", collapse=','), "), #,'draw'
        options=leaflet::layersControlOptions(collapsed=TRUE)
      ) %>%
      leaflet::hideGroup(group=c(", paste0("'", extract_period, "'", collapse=','), ")) %>%
      ", time_leaf, ";
      leaf"
  )
  
  leaf <- eval(parse(text = c))
  return(leaf)
}


#' explore
#' @title Leaflet map of present and paleo configurations for biogeographic systems
#' 
#' @description generate a timelapse or exploration visualisation for a reconstructed biomes 
#' 
#' @param x tabs. Object of class tabs, after running the reconstruct-function.
#' @param timelapse integer, specifies the speed of the html-animation, the higher the number the slower the animation 
#' @param filename name of the file to save
#'
#' @return html file
#'
#' @export
#'
#' 
#' @inherit reconstruct examples
#' 
explore <- function(x,
                    timelapse=NULL, #TODO add option to edit (select reference polygons that are missing and update the topo - set pixel(s) to 1 m above sealevel)
                    filename=NULL){
    # import, if needed
    if(base::is.character(x)){
      if(base::grepl('.rds$|.qs2$',x)){
        x <- import(x)
      }
    }
  
  # EXPLORE
  if(!base::is.null(timelapse)){
    
    # FALSE or 0 means exploratory map
    if (base::isFALSE(timelapse) || 
        base::identical(timelapse, 0L) || 
        base::identical(timelapse, 0)) {
      timelapse <- NULL
    }
    
  }
  
  if(!base::is.null(timelapse)){
    
    # TRUE default speed (1)
    if (base::isTRUE(timelapse)) timelapse <- 1
    
    # speed scale 
    speed <- suppressWarnings(as.numeric(timelapse))
    if (is.na(speed)) speed <- 1
    speed <- max(1, min(10, speed))
    
    leaf <- timelapse(paleostack = x$recvect, 
                      overlay = 2,
                      speed = speed)
    
  } else {

    leaf  <- exploring(labs = x$labs, 
                       aslv = x$recvect[[1]], 
                       simplify = FALSE)
    
  }
    
    # export
    if(!is.null(filename)){
      saveWidget(widget=leaf,filename)
    }
    
  return(leaf)
  }


#' @title reclass_names
#' 
#' @description reclass the unnamed / unknown names 
#' 
#' @param islandnames island names 
#'
#' @return style class
#' 
#'
#' @noRd
#' @keywords internal
#'
#'
reclass_names <- function(islandnames){
  style <- data.frame(name=islandnames,class='NAMED')
  style[grepl('^I-',style$name),'class'] <- 'I'
  style[grepl('^N-',style$name),'class'] <- 'N'
  style[grepl('^UNNAMED',style$name),'class'] <- 'UNNAMED'
  style[grepl('^UNKNOWN',style$name),'class'] <- 'UNKNOWN'
  return(style)
}






