% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/weight-association.R
\name{wa_test}
\alias{wa_test}
\alias{print.wa_test}
\alias{summary.wa_test}
\alias{tidy.wa_test}
\alias{glance.wa_test}
\title{Weight-Association Tests for Survey Weights}
\usage{
wa_test(
  model,
  type = c("DD", "PS1", "PS1q", "PS2", "PS2q", "WF"),
  coef_subset = NULL,
  aux_design = NULL,
  na.action = stats::na.omit
)

\method{print}{wa_test}(x, ...)

\method{summary}{wa_test}(object, ...)

\method{tidy}{wa_test}(x, ...)

\method{glance}{wa_test}(x, ...)
}
\arguments{
\item{model}{An object of class \code{svyglm}.}

\item{type}{Character string specifying the test type:
\code{"DD"}, \code{"PS1"}, \code{"PS1q"}, \code{"PS2"}, \code{"PS2q"}, \code{"WF"}.}

\item{coef_subset}{Optional character vector of coefficient names to include
in the test. Defaults to all coefficients.}

\item{aux_design}{Optional matrix or function to generate auxiliary regressors
for PS1/PS2 tests. If a function, it should take \code{X} and \code{y}
and return a matrix of extra columns to include.}

\item{na.action}{Function to handle missing data before testing.}

\item{x}{An object of class wa_test}

\item{...}{Additional arguments passed to methods}

\item{object}{An object of class wa_test}
}
\value{
An object of class \code{"wa_test"} containing:
  \item{statistic}{F-test statistic}
  \item{parameter}{Degrees of freedom (numerator, denominator)}
  \item{p.value}{P-value for the test}
  \item{method}{Name of the test performed}
  \item{call}{Function call}
}
\description{
Implements several weight-association tests that examine whether survey
weights are informative about the response variable after conditioning
on covariates. Variants include DuMouchel-Duncan (DD), Pfeffermann-Sverchkov
(PS1 and PS2, with optional quadratic terms or user-supplied auxiliary designs),
and Wu-Fuller (WF).
}
\details{
Let \eqn{y} denote the response, \eqn{X} the design matrix of covariates,
and \eqn{w} the survey weights. The null hypothesis in all cases is that
the weights are \emph{non-informative} given \eqn{X}, i.e. they do not
provide additional information about \eqn{y} beyond the covariates.

The following test variants are implemented:

\itemize{

  \item \strong{DuMouchel–Duncan (DD)}:
    After fitting the unweighted regression
    \deqn{\hat\beta = (X^\top X)^{-1} X^\top y,}
    compute residuals \eqn{e = y - X\hat\beta}.
    The DD test regresses \eqn{e} on the weights \eqn{w}:
    \deqn{e = \gamma_0 + \gamma_1 w + u.}
    A significant \eqn{\gamma_1} indicates association between weights
    and residuals, hence informativeness.

  \item \strong{Pfeffermann–Sverchkov PS1}:
    Augments the outcome regression with functions of the weights as
    auxiliary regressors:
    \deqn{y = X\beta + f(w)\theta + \varepsilon.}
    Under the null, \eqn{\theta = 0}. Quadratic terms
    (\eqn{w^2}) can be included (\code{"PS1q"}), or the user may supply
    a custom auxiliary design matrix \eqn{f(w)}.

  \item \strong{Pfeffermann–Sverchkov PS2}:
    First regress the weights on the covariates,
    \deqn{w = X\alpha + \eta,}
    and obtain fitted values \eqn{\hat w}.
    Then augment the outcome regression with \eqn{\hat w} (and optionally
    \eqn{\hat w^2} for \code{"PS2q"}):
    \deqn{y = X\beta + g(\hat w)\theta + \varepsilon.}
    Again, \eqn{\theta = 0} under the null.

  \item \strong{Wu–Fuller (WF)}:
    Compares weighted and unweighted regression fits. Let
    \eqn{\hat\beta_W} and \eqn{\hat\beta_U} denote the weighted and
    unweighted estimators. The test statistic is based on
    \deqn{T = (\hat\beta_W - \hat\beta_U)^\top
               \widehat{\mathrm{Var}}^{-1}(\hat\beta_W - \hat\beta_U)
    }
    and follows an approximate \eqn{F} distribution. A large value
    indicates that weights materially affect the regression.

}

In all cases, the reported statistic is an \eqn{F}-test with numerator
degrees of freedom equal to the number of auxiliary regressors added,
and denominator degrees of freedom equal to the residual degrees of
freedom from the augmented regression.
}
\examples{
# Load in survey package (required) and load in example data
library(survey)
data(api, package = "survey")

# Create a survey design and fit a weighted regression model
des <- svydesign(id = ~1, strata = ~stype, weights = ~pw, data = apistrat)
fit <- svyglm(api00 ~ ell + meals, design = des)

# Run weight-association diagnostic test; reports F-stat, df's, and p-value
results <- wa_test(fit, type = "DD")
print(results)

}
\references{
DuMouchel, W. H., & Duncan, G. J. (1983).
  Using sample survey weights in multiple regression analyses of stratified samples.
  *Journal of the American Statistical Association*, 78(383), 535-543.

Pfeffermann, D., & Sverchkov, M. (1999).
  Parametric and semi-parametric estimation of regression models fitted to survey data.
  *Sankhya: The Indian Journal of Statistics, Series B*, 61(1), 166-186.

Pfeffermann, D., & Sverchkov, M. (2003).
  Fitting generalized linear models under informative sampling.
  In R. L. Chambers & C. J. Skinner (Eds.), *Analysis of Survey Data*
  (pp. 175-196). Wiley.

Wu, Y., & Fuller, W. A. (2005).
  Preliminary testing procedures for regression with survey samples.
  In *Proceedings of the Joint Statistical Meetings, Survey Research Methods Section*
  (pp. 3683-3688). American Statistical Association.
}
\seealso{
\code{\link{diff_in_coef_test}} for the Hausman-Pfeffermann difference-in-coefficients test,
and \code{\link{svytestCE}} for the example dataset included in this package.
}
