% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/difference-in-coefficients.R
\name{diff_in_coef_test}
\alias{diff_in_coef_test}
\alias{print.diff_in_coef_test}
\alias{summary.diff_in_coef_test}
\alias{tidy.diff_in_coef_test}
\alias{glance.diff_in_coef_test}
\title{Difference-in-Coefficients Test for Survey Weights}
\usage{
diff_in_coef_test(
  model,
  lower.tail = FALSE,
  var_equal = TRUE,
  robust_type = c("HC0", "HC1", "HC2", "HC3"),
  coef_subset = NULL,
  na.action = stats::na.omit
)

\method{print}{diff_in_coef_test}(x, ...)

\method{summary}{diff_in_coef_test}(object, ...)

\method{tidy}{diff_in_coef_test}(x, ...)

\method{glance}{diff_in_coef_test}(x, ...)
}
\arguments{
\item{model}{An object of class \code{svyglm}.}

\item{lower.tail}{Logical; passed to \code{pchisq()}.}

\item{var_equal}{Logical; assume equal residual variance between models.
If \code{FALSE}, a heteroskedasticity-robust variance estimator is used.}

\item{robust_type}{Character; type of heteroskedasticity-robust variance
estimator to use if \code{var_equal = FALSE}. Options are
\code{"HC0"}, \code{"HC1"}, \code{"HC2"}, \code{"HC3"} as used in `sandwich` package.}

\item{coef_subset}{Character vector of coefficient names to include in the test.
Defaults to all coefficients.}

\item{na.action}{Function to handle missing data before fitting the test.}

\item{x}{An object of class diff_in_coef_test}

\item{...}{Additional arguments passed to methods}

\item{object}{An object of class diff_in_coef_test}
}
\value{
An object of class \code{"diff_in_coef_test"} containing:
  \item{statistic}{Chi-squared test statistic}
  \item{parameter}{Degrees of freedom}
  \item{p.value}{P-value for the test}
  \item{betas_unweighted}{Unweighted coefficient estimates}
  \item{betas_weighted}{Weighted coefficient estimates}
  \item{vcov_diff}{Estimated variance-covariance matrix of coefficient differences}
  \item{diff_betas}{Vector of coefficient differences}
  \item{call}{Function call}
}
\description{
Implements the Hausman-Pfeffermann Difference-in-Coefficients test
to assess whether survey weights significantly affect regression estimates.
}
\details{
Let \eqn{X} denote the design matrix and \eqn{y} the response vector.
Define the unweighted OLS estimator
\deqn{\hat\beta_{U} = (X^\top X)^{-1} X^\top y,}
and the survey-weighted estimator
\deqn{\hat\beta_{W} = (X^\top W X)^{-1} X^\top W y,}
where \eqn{W = \mathrm{diag}(w_1, \ldots, w_n)} is the diagonal matrix of survey weights.

The test statistic is based on the difference
\deqn{d = \hat\beta_{W} - \hat\beta_{U}.}

Under the null hypothesis that weights are not informative,
\eqn{d} has mean zero and variance \eqn{V_d}.
The test statistic is
\deqn{T = d^\top V_d^{-1} d,}
which is asymptotically \eqn{\chi^2_p} distributed with
\eqn{p} equal to the number of coefficients tested.

If \code{var_equal = TRUE}, \eqn{V_d} is estimated assuming equal residual variance
across weighted and unweighted models. If \code{var_equal = FALSE}, a
heteroskedasticity-robust estimator (e.g. HC0–HC3) is used.

This test is a survey-weighted adaptation of the Hausman specification test
(Hausman, 1978), as proposed by Pfeffermann (1993).
}
\examples{
# Load in survey package (required) and load in example data
library(survey)
data(api, package = "survey")

# Create a survey design and fit a weighted regression model
des <- svydesign(id = ~1, strata = ~stype, weights = ~pw, data = apistrat)
fit <- svyglm(api00 ~ ell + meals, design = des)

# Run difference-in-coefficients diagnostic test versions with different variance assumptions
# and reports Chi-Squared statistic, df, and p-value
summary(diff_in_coef_test(fit, var_equal = TRUE))
summary(diff_in_coef_test(fit, var_equal = FALSE, robust_type = "HC3"))

}
\references{
Hausman, J. A. (1978). Specification Tests in Econometrics.
  *Econometrica*, 46(6), 1251-1271. \doi{10.2307/1913827}

Pfeffermann, D. (1993). The Role of Sampling Weights When Modeling Survey Data.
  *International Statistical Review*, 61(2), 317-337. \doi{10.2307/1403631}
}
\seealso{
\code{\link{svytestCE}} for the curated Consumer Expenditure dataset
  included in this package, which can be used to demonstrate the
  Difference-in-Coefficients test.

  @importFrom survey svyglm
}
