\name{skeleton}
\alias{skeleton}
\alias{print.skeleton}
\title{Create a Translation Template for Sumerian Text}
\description{
Creates a structured template (skeleton) for translating Sumerian text. The template displays each word and syllable with its sign name and cuneiform representation, providing a framework for adding translations.

The function \code{skeleton} computes the template and returns an object of class \code{"skeleton"}. The \code{print} method displays the template in the console.
}
\usage{
skeleton(x, mapping = NULL)

\method{print}{skeleton}(x, ...)
}
\arguments{
  \item{x}{For \code{skeleton}: A character string of length 1 containing transliterated Sumerian text. Words are separated by spaces, syllables within words by hyphens or other separators.

  For \code{print.skeleton}: An object of class \code{"skeleton"} as returned by \code{skeleton}.}
  \item{mapping}{A data frame containing the sign mapping table with columns \code{syllables}, \code{name}, and \code{cuneiform}. If \code{NULL} (the default), the package's internal mapping file \file{etcsl_mapping.txt} is loaded.}
  \item{...}{Additional arguments passed to the print method (currently unused).}
}
\details{
The function generates a hierarchical template with different levels of detail depending on the input type:

\describe{
  \item{Multiple words}{The template includes a header line with the original text, followed by entries for each word, its syllables (indented with one tab), and sub-signs for multi-sign syllables (indented with two tabs).}
  \item{Single word (multiple syllables)}{The word equation serves as the header, followed by syllable entries (one tab) and sub-sign entries (two tabs). No redundant header line is generated.}
  \item{Single syllable}{Only the syllable equation is shown (no indentation), with sub-sign entries indented by one tab if applicable.}
}

Each line in the template follows the format

\code{|[tabs]reading=SIGN.NAME=cuneiform::}

The template should be filled in as follows:
\itemize{
  \item Between the two colons: the grammatical type of the expression (e.g., \code{S} for noun phrases, \code{V} for verbs, etc.). See \code{\link{make_dictionary}} for details.
  \item After the second colon: the translation
}
For example, a filled-in line might look like:

\code{|an=AN=<cuneiform sign for AN>:S: god of heaven}

Redundant lines are automatically omitted: if a word consists of only one syllable, no separate syllable line is generated.

This function is intended to be used together with \code{\link{look_up}} for translating Sumerian texts: first create a template with \code{skeleton}, then use \code{look_up} to find the meanings of words and signs, and fill in the template accordingly.

The template format is designed to be saved as a text file (.txt) or Word document (.docx), filled in manually, and can then be used as input for \code{\link{make_dictionary}} to create a custom dictionary.
}
\value{
\code{skeleton} returns a character vector of class \code{c("skeleton", "character")} containing the template lines.

\code{print.skeleton} prints the template to the console and returns \code{x} invisibly.
}
\seealso{
\code{\link{look_up}} for looking up translations of Sumerian signs and words,
\code{\link{make_dictionary}} for creating a dictionary from filled-in templates,
\code{\link{info}} for retrieving detailed sign information
}
\examples{
# Create a template for a multi-word phrase
skeleton("e-ta-na an-ce3 ba-ed3-de3")

# Create a template for a single word
skeleton("lugal-e")

# Create a template for a single syllable
skeleton("an")

# Store the template for further use
tmpl <- skeleton("lu2 du")
tmpl

# Typical workflow: create template, then look up meanings
dic <- read_dictionary()
tmpl <- skeleton("lugal kur-ra-ke4")
print(tmpl)
look_up("lugal", dic)
look_up("kur", dic)
}
\keyword{utilities}
\keyword{character}
