#' Calculate the Norm of a Numeric Vector
#'
#' This function calculates the norm of a numeric vector \code{u}.
#' It supports the standard Euclidean norm, a generalized version with a weight matrix,
#' or a weighted norm with a weight vector.
#'
#' @param u A numeric vector. The vector whose norm is to be calculated.
#' @param w Optional. Can be \code{NULL}, a square matrix, or a numeric vector:
#'
#' \item \code{NULL}: The standard Euclidean norm is calculated (\eqn{\sqrt{\sum u_i^2}}).
#' \item Square matrix: The generalized norm with a weight matrix is calculated (\eqn{\sqrt{u^T W u}}).
#' \item Numeric vector: A weighted norm is calculated (\eqn{\sqrt{\sum w_i u_i^2}}).
#'
#' @return A positive real number representing the norm of the vector \code{u}.
#'
#' @keywords internal
#' @noRd
.norm <- function(u, w = NULL) {
  # Ensure the input vector u is numeric
  stopifnot(is.numeric(u))

  # Coerce u explicitly to a numeric vector (protects against factors)
  u <- as.numeric(u)

  # ----------------------------------------------------------------------
  # CASE 1: No weights provided (w = NULL)
  # Computes the standard Euclidean norm: sqrt(u₁² + u₂² + ... + uₙ²)
  # ----------------------------------------------------------------------
  if (is.null(w)) return(sqrt(sum(u * u)))

  # ----------------------------------------------------------------------
  # CASE 2: Weighted norm using a weight matrix 'w'
  # The matrix must be square and have the same dimension as u.
  # Computes sqrt( uᵀ W u ), the quadratic form induced by W.
  # ----------------------------------------------------------------------
  if (is.matrix(w)) { stopifnot(ncol(w) == length(u), nrow(w) == length(u)); return(sqrt(drop(t(u) %*% w %*% u))) }

  # ----------------------------------------------------------------------
  # CASE 3: Weighted norm using a numeric vector of weights
  # Computes sqrt( Σ (wᵢ * uᵢ²) ), a diagonal-weighted norm.
  # Equivalent to sqrt( uᵀ diag(w) u ).
  # ----------------------------------------------------------------------
  if (is.numeric(w)) { stopifnot(length(w) == length(u)); return(sqrt(sum(w * (u * u)))) }

  # ----------------------------------------------------------------------
  # CASE 4: Invalid input for 'w'
  # Rejects any unsupported type to ensure mathematical consistency.
  # ----------------------------------------------------------------------
  stop("The second argument 'w' must be NULL, a square matrix, or a numeric vector.")
}

#' Normalize a Vector, Matrix, Data Frame, or List
#'
#' This function applies normalization (unit norm) to an object \code{x}, dividing it by its norm.
#' It can handle numeric vectors, matrices, data frames, or lists of these.
#' It uses the \code{\link{.norm}} function to compute the norm, allowing for optional weights.
#'
#' @param x Object to normalize. Can be:
#' \item A numeric vector (normalized to length 1)
#' \item A matrix or \code{data.frame} (each row is normalized)
#' \item A list (recursively applies to each element)
#'
#' @param w Optional. Weights for the normalization, passed to \code{\link{.norm}}:
#' \item \code{NULL}: standard Euclidean norm
#' \item Numeric vector: individual weights
#' \item Square matrix: generalized metric
#'
#' @return The normalized object \code{x}, with the same structure (vector, matrix, or list).
#'
#' @keywords internal
#' @noRd
normalizes <- function(x, w = NULL) {
  # ----------------------------------------------------------------------
  # CASE 1: x is a numeric vector
  # Performs vector normalization using the norm defined in .norm().
  # Returns the vector scaled so that its norm equals 1.
  # ----------------------------------------------------------------------
  if (is.numeric(x) && is.null(dim(x))) {
    nrm <- .norm(x, w)

    # If the norm is (numerically) zero, normalization is impossible.
    # Return x unchanged to avoid dividing by zero.
    if (isTRUE(all.equal(nrm, 0))) return(x)
    return(x / nrm)
  }

  # ----------------------------------------------------------------------
  # CASE 2: x is a matrix or a data frame
  # Normalization is applied **row-wise**.
  # For PCA/STATIS, each row typically represents an observation.
  # ----------------------------------------------------------------------
  if (is.matrix(x) || is.data.frame(x)) {
    # Convert data frame to numeric matrix if necessary
    X <- if (is.data.frame(x)) as.matrix(x) else x

    # Apply normalization to each row independently
    out <- t(apply(X, 1, function(r) { nrm <- .norm(r, w); if (isTRUE(all.equal(nrm, 0))) r else r / nrm }))

    # Preserve row and column names from the original matrix
    rownames(out) <- rownames(X); colnames(out) <- colnames(X); return(out)
  }

  # ----------------------------------------------------------------------
  # CASE 3: x is a list
  # Apply normalization recursively to each element.
  # Allows normalizing lists of vectors or matrices.
  # ----------------------------------------------------------------------
  if (is.list(x)) return(lapply(x, function(elem) normalizes(elem, w)))

  # ----------------------------------------------------------------------
  # CASE 4: Unsupported type
  # The function handles vectors, matrices/data frames, and lists.
  # Any other type triggers an informative error.
  # ----------------------------------------------------------------------
  stop("Type of 'x' not supported.")
}

#' Select and prepare a subset of variables from a supervision matrix
#'
#' This function selects a predefined subset of variables (\code{ETCal}) from a
#' supervision matrix (\code{superv}), checks dimension consistency, verifies
#' missing variables, and constructs a clean data frame containing the first two
#' coordinates typically used for PCA or STATIS correlation plots.
#'
#' @param superv A numeric matrix or data frame where each row corresponds to a variable and columns represent coordinates (\code{res$supervariables}).
#' @param ET A character vector containing the full list of expected variable names (\code{res$vars.names}).
#' @param ETCal A character vector containing the subset of variables to be selected.
#'
#' @return A data frame with two columns:
#'   \itemize{
#'     \item \code{x}: first coordinate (e.g., PC1)
#'     \item \code{y}: second coordinate (e.g., PC2)
#'   }
#'   Row names correspond to the selected variables defined in \code{ETCal}.
#'
#' @details
#' The function performs the following steps:
#' \enumerate{
#'   \item Checks that the number of rows in \code{superv} matches the length of \code{ET}.
#'   \item Assigns the row names of \code{superv} using \code{ET}.
#'   \item Identifies whether any variables in \code{ETCal} are missing in \code{superv};
#'         missing variables trigger a warning.
#'   \item Creates an ordered list of valid variables (\code{ETCal_ok}) based on their presence in \code{superv}.
#'   \item Extracts the corresponding rows from \code{superv} and constructs a data frame with columns \code{x} and \code{y}.
#' }
#'
#' @examples
#' data(Tuis5_95, Tuis5_96, Tuis5_97, Tuis5_98)
#' labels <- c("95","96","97","98")
#'
#' res <- statis.dual(list(Tuis5_95, Tuis5_96, Tuis5_97, Tuis5_98), labels.tables = labels)
#'
#' ETCal <- c("Ph","Temp","DBO","ST","PO4","NO3","POD","Cal")
#'
#' df_selected <- select.super.variables(res$supervariables, res$vars.names, ETCal)
#'
#' @export
select.super.variables <- function(superv, ET, ETCal) {

  # Check dimensions
  stopifnot(nrow(superv) == length(ET))

  # Assign row names
  rownames(superv) <- ET

  # Identify missing variables
  missing_vars <- setdiff(ETCal, rownames(superv))
  if (length(missing_vars) > 0) {
    warning(sprintf(
      "Variables not found in 'superv': %s",
      paste(missing_vars, collapse = ", ")
    ))
  }

  # Variables present, keeping ETCal order
  ETCal_ok <- intersect(ETCal, rownames(superv))

  # Select rows
  superv_sel <- superv[ETCal_ok, , drop = FALSE]

  # Output clean data frame
  superv_sel_df <- data.frame(
    x = superv_sel[, 1],
    y = superv_sel[, 2],
    row.names = ETCal_ok
  )

  return(superv_sel_df)
}
