% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/slp_generate.R
\name{slp_generate}
\alias{slp_generate}
\title{Generate a Standard Load Profile}
\source{
\url{https://www.bdew.de/energie/standardlastprofile-strom/}

\url{https://www.bdew.de/media/documents/1999_Repraesentative-VDEW-Lastprofile.pdf}

\url{https://www.bdew.de/media/documents/2000131_Anwendung-repraesentativen_Lastprofile-Step-by-step.pdf}
}
\usage{
slp_generate(profile_id, start_date, end_date, state_code = NULL)
}
\arguments{
\item{profile_id}{load profile identifier, required}

\item{start_date}{start date in ISO 8601 format, required}

\item{end_date}{end date in ISO 8601 format, required}

\item{state_code}{identifier for one of 16 German states, optional}
}
\value{
A data.frame with four variables:
\itemize{
\item \code{profile_id}, character, load profile identifier
\item \code{start_time}, POSIXct / POSIXlt, start time
\item \code{end_time}, POSIXct / POSIXlt, end time
\item \code{watts}, numeric, electric power
}
}
\description{
Generate a standard load profile, normalized to an annual
consumption of 1,000 kWh.
}
\details{
In regards to the electricity market in Germany, the term "Standard Load
Profile" refers to a representative pattern of electricity consumption over
a specific period. These profiles can be used to depict the expected electricity
consumption for various customer groups, such as households or businesses.

For each distinct combination of \code{profile_id}, \code{period}, and \code{day}, there
are 96 x 1/4 hour measurements of electrical power. Values are normalized so
that they correspond to an annual consumption of 1,000 kWh. That is, summing
up all the quarter-hourly consumption values for one year yields an approximate
total of 1,000 kWh/a; for more information, refer to the 'Examples' section,
or call \code{vignette("algorithm-step-by-step")}.

In total there are 11 \code{profile_id} for three different customer groups:
\itemize{
\item Households: \code{H0}
\item Commercial: \code{G0}, \code{G1}, \code{G2}, \code{G3}, \code{G4}, \code{G5}, \code{G6}
\item Agriculture: \code{L0}, \code{L1}, \code{L2}
}

For more information and examples, call \code{\link[=slp_info]{slp_info()}}.

Period definitions:
\itemize{
\item \code{summer}: May 15 to September 14
\item \code{winter}: November 1 to March 20
\item \code{transition}: March 21 to May 14, and September 15 to October 31
}

Day definitions:
\itemize{
\item \code{workday}: Monday to Friday
\item \code{saturday}: Saturdays; Dec 24th and Dec 31th are considered a Saturdays too
if they are not a Sunday
\item \code{sunday}: Sundays and all public holidays
}

\strong{Note}: The package supports public holidays for Germany,
retrieved from the \href{https://github.com/nager/Nager.Date}{nager.Date API}.
Use the optional argument \code{state_code} to consider public holidays on a state
level too. Allowed values are listed below:
\itemize{
\item \code{DE-BB}: Brandenburg
\item \code{DE-BE}: Berlin
\item \code{DE-BW}: Baden-Württemberg
\item \code{DE-BY}: Bavaria
\item \code{DE-HB}: Bremen
\item \code{DE-HE}: Hesse
\item \code{DE-HH}: Hamburg
\item \code{DE-MV}: Mecklenburg-Vorpommern
\item \code{DE-NI}: Lower-Saxony
\item \code{DE-NW}: North Rhine-Westphalia
\item \code{DE-RP}: Rhineland-Palatinate
\item \code{DE-SH}: Schleswig-Holstein
\item \code{DE-SL}: Saarland
\item \code{DE-SN}: Saxony
\item \code{DE-ST}: Saxony-Anhalt
\item \code{DE-TH}: Thuringia
}

\code{start_date} must be greater or equal to "1990-01-01". This is because public
holidays in Germany would be ambitious before the reunification in 1990
(think of the state of Berlin in 1989 and earlier).

\code{end_date} must be smaller or equal to "2073-12-31" because this is last
year supported by the \href{https://github.com/nager/Nager.Date}{nager.Date API}.
}
\examples{
start <- "2024-01-01"
end <- "2024-12-31"

# multiple profile IDs are supported
L <- slp_generate(c("L0", "L1", "L2"), start, end)
head(L)

# you can specify one of the 16 ISO 3166-2:DE codes to take into
# account holidays determined at the level of the federal states
berlin <- slp_generate("H0", start, end, state_code = "DE-BE")

# for convenience, the codes can be specified without the prefix "DE-"
identical(berlin, slp_generate("H0", start, end, state_code = "BE"))

# state codes are not case-sensitive
identical(berlin, slp_generate("H0", start, end, state_code = "de-be"))

# consider only nationwide public holidays
H0_2024 <- slp_generate("H0", start, end)

# electric power values are normalized to consumption of ~1,000 kWh/a
sum(H0_2024$watts / 4 / 1000)

}
