% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/HIP.R
\name{HIP}
\alias{HIP}
\title{Halton Iterative Partitioning (HIP).}
\usage{
HIP(
  population = NULL,
  n = 20,
  iterations = 7,
  minRadius = NULL,
  panels = NULL,
  panel_overlap = NULL,
  verbose = FALSE
)
}
\arguments{
\item{population}{A population of point pairs.}

\item{n}{The number of points to draw from the population. Default 20.}

\item{iterations}{The levels of partitioning required. Default 7.}

\item{minRadius}{If specified, the minimum distance, in meters, allowed between sample
points. This is applied to the $overSample.}

\item{panels}{A list of integers that define the size of each panel in a
non-overlapping panels design. The length of the list determines the number of
panels required. The sum of the integers in the panels parameter will determine
the total number of samples selected, n. The default value for panels is NULL,
this indicates that a non-overlapping panel design is not wanted.}

\item{panel_overlap}{A list of integers that define the overlap into the previous
panel. Is only used when the panels parameter is not NULL. The default value for
panel_overlap is NULL. The length of panel_overlap must be equal to the length
of panels. The first value is always forced to zero as the first panel never
overlaps any region.}

\item{verbose}{Boolean if you want to see any output printed to screen. Helpful if taking a
long time. Default is FALSE i.e. no informational messages are displayed.}
}
\value{
Return a list containing the following five variables:

\itemize{
\item \code{Population} Original population point pairs as an sf object.
\item \code{HaltonIndex} The Halton index for the point. Points will be spread equally across
all Halton indices.
\item \code{sample} The sample from the population of size n.
\item \code{overSample} The overSample contains one point from each Halton box. All contiguous
sub-samples from oversample are spatially balanced, and the first n points are identical to sample.
\item \code{minRadius} This result variable contains the sample created using the minRadius
parameter. If the minRadius parameter is not specified then the minRadius variable will contain NULL.
}
}
\description{
HIP draws spatially balanced samples and over-samples from point
resources by partitioning the resource into boxes with the same nested structure
as Halton boxes. The \strong{spbal} parameter \strong{iterations} defines the number of
boxes used in the HIP partition and should be larger than the sample size but
less than the population size. The \strong{iterations parameter} also defines the
number of units available in the HIP over-sample, where the over-sample contains
one unit from each box in the HIP partition.
}
\details{
Halton iterative partitioning (HIP) extends Basic acceptance
sampling (BAS) to point resources. It partitions the resource into $B >= n$
boxes that have the same nested structure as in BAS, but different sizes.
These boxes are then uniquely numbered using a random-start Halton sequence
of length $B$. The HIP sample is obtained by randomly drawing one point from
each of the boxes numbered $1, 2, . . . , n$.
}
\examples{
# generating 20 points from a population of 5,000 (random) points with 7
# levels of partitioning (4 in the first dimension and 3 in the second) to
# give (2^4) * (3^3) = 32 * 27, resulting in 864 boxes ----------------------

# set random seed
set.seed(511)

# define HIP parameters.
pop <- matrix(runif(5000*2), nrow = 5000, ncol = 2)
n <- 20
its <- 7

# Convert the population matrix to an sf point object.
sf_points <- sf::st_as_sf(data.frame(pop), coords = c("X1", "X2"))
dim(sf::st_coordinates(sf_points))

# generate HIP sample.
result <- spbal::HIP(population = sf_points,
                     n = n,
                     iterations =  its)

# HaltonIndex
HaltonIndex <- result$HaltonIndex
table(HaltonIndex)

# Population Sample
HIPsample <- result$sample
HIPsample

}
\author{
Phil Davies, Blair Robertson.
}
