% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/makeNKFunction.R
\name{makeMNKFunction}
\alias{makeMNKFunction}
\alias{makeRMNKFunction}
\title{Generators for (r)MNK-landscapes}
\usage{
makeMNKFunction(M, N, K, funs = NULL)

makeRMNKFunction(M, N, K, rho = 0)
}
\arguments{
\item{M}{[\code{integer(1)}]\cr
Number of objectives (at least two).}

\item{N}{[\code{integer(1)}]\cr
Length of the bit-string (decision space dimension).}

\item{K}{[\code{integer}]\cr
Epistatic links. Possible values are a a) single integer which is used for all
bits and positions, b) an integer vector of \code{N} integers that are used
across the objectives, c) a list of \code{M} length-\code{N} integers to
define specific epistatic links for every bit position of every objective.}

\item{funs}{[\code{list} | \code{NULL}]\cr
Allows for an alternative way to build a MNK-landscape by passing a list of
at least two single-objective NK-landscapes. In this case all other parameters
\code{M}, \code{N}, \code{K} and \code{rho} are ignored. Note that the passed
functions must be compatible, i.e., a) the input dimension \code{N} needs to match
and b) all pased functions need to be multi-objective.
Default is \code{NULL}.}

\item{rho}{[\code{numeric(1)}]\cr
Correlation between objectives (value between -1 and 1).}
}
\value{
[\code{smoof_multi_objective_function}]
}
\description{
Generators for multi-objective NK-landscapes, i.e., with at least two objectives.
Function \code{makeMNKLandscape(M, N, K)} create NK-landscapes with \code{M}
(\eqn{\geq 2}) objectives, input dimension \code{N}, and epistatic links specified
via parameter \code{K}. \code{K} can be a single integer value, a vector of
\code{M} integers or a list of length-\code{N} integer vectors (see parameter
description for details) which allow for maximum flexibility. It is also
possible to compose a MNK-landscape by passing a list of single-objective
NK-landscapes via argument \code{funs}.
}
\details{
Function \code{makeRMNKLandscape(M, N, K, rho)} generates a MNK-landscape
with correlated objective function values. The correlation can be adjusted
by setting the \code{rho} parameter to a value between minus one and one.
}
\examples{
# generate homogeneous uncorrelated bi-objective MNK-landscape with each
# three epistatic links
M = 2L
N = 20L
K = 3L
fn = makeMNKFunction(M, N, K)

# generate MNK-landscape where the first function has 3 epistatic links
# per bit while the second function has 2
fn = makeMNKFunction(M, N, K = c(3L, 2L))

# sample the number of epistatic links individually from {1, ..., 5} for
# every bit position and every objective
K = lapply(seq_len(M), function(m) sample(1:(N-1), size = N, replace = TRUE))
fn = makeMNKFunction(M, N, K = K)

#' # generate strongly positively correlated objectives
fn = makeRMNKFunction(M, N, K, rho = 0.9)

# alternative constructor: generate two single-objective NK-landscapes
# and combine into bi-objective problem
soofn1 = makeNKFunction(N, K = 2L) # homegeneous in K
K = sample(2:3, size = N, replace = TRUE)
soofn2 = makeNKFunction(N, K = K) # heterogeneous in K
moofn = makeMNKFunction(funs = list(soofn1, soofn2))
getNumberOfObjectives(moofn)

}
\references{
H. E. Aguirre and K. Tanaka, Insights on properties of multiobjective MNK-landscapes,
Proceedings of the 2004 Congress on Evolutionary Computation, Portland, OR, USA, 2004,
pp. 196-203 Vol.1, doi: 10.1109/CEC.2004.1330857.
}
\seealso{
\code{\link{makeNKFunction}}

Other nk_landscapes: 
\code{\link{exportNKFunction}()},
\code{\link{makeNKFunction}()}
}
\concept{nk_landscapes}
