% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/semidist.R
\name{sd_test}
\alias{sd_test}
\title{Semi-distance independence test}
\usage{
sd_test(X, y, test_type = "perm", num_perm = 10000)
}
\arguments{
\item{X}{Data of multivariate continuous variables, which should be an
\eqn{n}-by-\eqn{p} matrix, or, a vector of length \eqn{n} (for univariate
variable).}

\item{y}{Data of categorical variables, which should be a factor of length
\eqn{n}.}

\item{test_type}{Type of the test:
\itemize{
\item \code{"perm"} (the default): Implement the test via permutation test;
\item \code{"asym"}: Implement the test via the asymptotic approximation when the
dimension of continuous variables \eqn{p} is high.
}

See the Reference for details.}

\item{num_perm}{The number of replications in permutation test. Defaults to
10000. See Details and Reference.}
}
\value{
A list with class \code{"indtest"} containing the following components
\itemize{
\item \code{method}: name of the test;
\item \code{name_data}: names of the \code{X} and \code{y};
\item \code{n}: sample size of the data;
\item \code{test_type}: type of the test;
\item \code{num_perm}: number of replications in permutation test, if
\code{test_type = "perm"};
\item \code{stat}: test statistic;
\item \code{pvalue}: computed p-value.
}
}
\description{
Implement the semi-distance independence test via permutation
test, or via the asymptotic approximation when the dimensionality of
continuous variables \eqn{p} is high.
}
\details{
The semi-distance independence test statistic is \deqn{T_n = n \cdot
  \widetilde{\text{SDcov}}_n(X, y),} where the
\eqn{\widetilde{\text{SDcov}}_n(X, y)} can be computed by \code{sdcov(X, y, type = "U")}.

For the permutation test (\code{test_type = "perm"}), totally \eqn{K}
replications of permutation will be conducted, and the argument \code{num_perm}
specifies the \eqn{K} here. The p-value of permutation test is computed by
\deqn{\text{p-value} = (\sum_{k=1}^K I(T^{\ast (k)}_{n} \ge T_{n}) + 1) /
  (K + 1),} where \eqn{T_{n}} is the semi-distance test statistic and
\eqn{T^{\ast (k)}_{n}} is the test statistic with \eqn{k}-th permutation
sample.

When the dimension of the continuous variables is high, the asymptotic
approximation approach can be applied (\code{test_type = "asym"}), which is
computationally faster since no permutation is needed.
}
\examples{
X <- mtcars[, c("mpg", "disp", "drat", "wt")]
y <- factor(mtcars[, "am"])
test <- sd_test(X, y)
print(test)

# Man-made independent data -------------------------------------------------
n <- 30; R <- 5; p <- 3; prob <- rep(1/R, R)
X <- matrix(rnorm(n*p), n, p)
y <- factor(sample(1:R, size = n, replace = TRUE, prob = prob), levels = 1:R)
test <- sd_test(X, y)
print(test)

# Man-made functionally dependent data --------------------------------------
n <- 30; R <- 3; p <- 3
X <- matrix(0, n, p)
X[1:10, 1] <- 1; X[11:20, 2] <- 1; X[21:30, 3] <- 1
y <- factor(rep(1:3, each = 10))
test <- sd_test(X, y)
print(test)

#' Man-made high-dimensionally independent data -----------------------------
n <- 30; R <- 3; p <- 100
X <- matrix(rnorm(n*p), n, p)
y <- factor(rep(1:3, each = 10))
test <- sd_test(X, y)
print(test)

test <- sd_test(X, y, test_type = "asym")
print(test)

# Man-made high-dimensionally dependent data --------------------------------
n <- 30; R <- 3; p <- 100
X <- matrix(0, n, p)
X[1:10, 1] <- 1; X[11:20, 2] <- 1; X[21:30, 3] <- 1
y <- factor(rep(1:3, each = 10))
test <- sd_test(X, y)
print(test)

test <- sd_test(X, y, test_type = "asym")
print(test)

}
\seealso{
\code{\link[=sdcov]{sdcov()}} for computing the statistic of semi-distance covariance.
}
