% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/marker_indices.R
\name{lmsi}
\alias{lmsi}
\title{Linear Marker Selection Index (LMSI)}
\usage{
lmsi(
  phen_mat = NULL,
  marker_scores = NULL,
  pmat,
  gmat,
  G_s = NULL,
  wmat,
  wcol = 1,
  selection_intensity = 2.063,
  GAY = NULL
)
}
\arguments{
\item{phen_mat}{Matrix of phenotypes (n_genotypes x n_traits).
Can be NULL if G_s is provided directly (theoretical case where
covariance structure is known without needing empirical data).}

\item{marker_scores}{Matrix of marker scores (n_genotypes x n_traits).
These are computed as s_j = sum(x_jk * beta_jk) where x_jk is the coded
marker value and beta_jk is the estimated marker effect for trait j.
Can be NULL if G_s is provided directly.}

\item{pmat}{Phenotypic variance-covariance matrix (n_traits x n_traits).}

\item{gmat}{Genotypic variance-covariance matrix (n_traits x n_traits).}

\item{G_s}{Genomic covariance matrix explained by markers (n_traits x n_traits).
This represents Var(s) which approximates Cov(y, s) when markers fully
explain genetic variance. If provided, phen_mat and marker_scores become
optional as the covariance structure is specified directly.
If NULL, computed empirically from marker_scores and phen_mat.}

\item{wmat}{Economic weights matrix (n_traits x k), or vector.}

\item{wcol}{Weight column to use if wmat has multiple columns (default: 1).}

\item{selection_intensity}{Selection intensity k (default: 2.063 for 10\% selection).}

\item{GAY}{Optional. Genetic advance of comparative trait for PRE calculation.}
}
\value{
List of class \code{"lmsi"} with components:
\describe{
  \item{\code{b_y}}{Coefficients for phenotypes (n_traits vector).}
  \item{\code{b_s}}{Coefficients for marker scores (n_traits vector).}
  \item{\code{b_combined}}{Combined coefficient vector [b_y; b_s] (2*n_traits vector).}
  \item{\code{P_L}}{Combined phenotypic-marker covariance matrix (2*n_traits x 2*n_traits).}
  \item{\code{G_L}}{Combined genetic-marker covariance matrix (2*n_traits x n_traits).}
  \item{\code{G_s}}{Genomic covariance matrix explained by markers (n_traits x n_traits).}
  \item{\code{rHI}}{Index accuracy (correlation between index and breeding objective).}
  \item{\code{sigma_I}}{Standard deviation of the index.}
  \item{\code{R}}{Selection response (k * sigma_I).}
  \item{\code{Delta_H}}{Expected genetic gain per trait (vector of length n_traits).}
  \item{\code{GA}}{Overall genetic advance in breeding objective.}
  \item{\code{PRE}}{Percent relative efficiency (if GAY provided).}
  \item{\code{hI2}}{Index heritability.}
  \item{\code{summary}}{Data frame with coefficients and metrics (combined view).}
  \item{\code{phenotype_coeffs}}{Data frame with phenotype coefficients only.}
  \item{\code{marker_coeffs}}{Data frame with marker score coefficients only.}
  \item{\code{coeff_analysis}}{Data frame with coefficient distribution analysis.}
}
}
\description{
Implements the LMSI which combines phenotypic information with molecular
marker scores from statistically significant markers (Lande & Thompson, 1990).
The index is I = b_y' * y + b_s' * s, where y are phenotypes and s are
marker scores.
}
\details{
\strong{Mathematical Formulation:}

The LMSI maximizes the correlation between the index
\eqn{I_{LMSI} = \mathbf{b}_y^{\prime}\mathbf{y} + \mathbf{b}_s^{\prime}\mathbf{s}}
and the breeding objective \eqn{H = \mathbf{w}^{\prime}\mathbf{g}}.

Combined covariance matrices:
\deqn{\mathbf{P}_L = \begin{bmatrix} \mathbf{P} & \text{Cov}(\mathbf{y}, \mathbf{s}) \\ \text{Cov}(\mathbf{y}, \mathbf{s})^{\prime} & \text{Var}(\mathbf{s}) \end{bmatrix}}
\deqn{\mathbf{G}_L = \begin{bmatrix} \mathbf{G} \\ \mathbf{G}_s \end{bmatrix}}

where \eqn{\mathbf{P}} is the phenotypic variance, \eqn{\text{Cov}(\mathbf{y}, \mathbf{s})}
is the covariance between phenotypes and marker scores (computed from data),
\eqn{\text{Var}(\mathbf{s})} is the variance of marker scores, \eqn{\mathbf{G}} is the
genotypic variance, and \eqn{\mathbf{G}_s} represents the genetic covariance
explained by markers.

Index coefficients:
\deqn{\mathbf{b}_{LMSI} = \mathbf{P}_L^{-1} \mathbf{G}_L \mathbf{w}}

Accuracy:
\deqn{\rho_{HI} = \sqrt{\frac{\mathbf{b}_{LMSI}^{\prime} \mathbf{G}_L \mathbf{w}}{\mathbf{w}^{\prime} \mathbf{G} \mathbf{w}}}}

Selection response:
\deqn{R_{LMSI} = k \sigma_{I_{LMSI}} = k \sqrt{\mathbf{b}_{LMSI}^{\prime} \mathbf{P}_L \mathbf{b}_{LMSI}}}

Expected genetic gain per trait:
\deqn{\mathbf{E}_{LMSI} = k \frac{\mathbf{G}_L^{\prime} \mathbf{b}_{LMSI}}{\sigma_{I_{LMSI}}}}
}
\examples{
\dontrun{
# Load data
data(seldata)
pmat <- phen_varcov(seldata[, 3:9], seldata[, 2], seldata[, 1])
gmat <- gen_varcov(seldata[, 3:9], seldata[, 2], seldata[, 1])

# Simulate marker scores (in practice, computed from QTL mapping)
set.seed(123)
n_genotypes <- 100
n_traits <- ncol(gmat)
marker_scores <- matrix(rnorm(n_genotypes * n_traits, mean = 5, sd = 1.5),
  nrow = n_genotypes, ncol = n_traits
)
colnames(marker_scores) <- colnames(gmat)

# Simulate phenotypes
phen_mat <- matrix(rnorm(n_genotypes * n_traits, mean = 15, sd = 3),
  nrow = n_genotypes, ncol = n_traits
)
colnames(phen_mat) <- colnames(gmat)

# Economic weights
weights <- c(10, 5, 3, 3, 5, 8, 4)

# Calculate LMSI
result <- lmsi(phen_mat, marker_scores, pmat, gmat,
  G_s = NULL, wmat = weights
)
print(result$summary)
}
}
\references{
Lande, R., & Thompson, R. (1990). Efficiency of marker-assisted selection
in the improvement of quantitative traits. Genetics, 124(3), 743-756.

Cerón-Rojas, J. J., & Crossa, J. (2018). Linear Selection Indices in Modern
Plant Breeding. Springer International Publishing. Chapter 4.
}
