% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rhierLinearMixtureParallel.R
\name{rhierLinearMixtureParallel}
\alias{rhierLinearMixtureParallel}
\title{MCMC Algorithm for Hierarchical Multinomial Linear Model with Mixture-of-Normals Heterogeneity}
\usage{
rhierLinearMixtureParallel(Data, Prior, Mcmc, verbose = FALSE)
}
\arguments{
\item{Data}{A list containing: `regdata` - A \eqn{nreg} size list of multinomial regdata, and optional `Z`- \eqn{nreg \times nz} matrix of unit characteristics.}

\item{Prior}{A list with one required parameter: `ncomp`, and optional parameters: `deltabar`, `Ad`, `mubar`, `Amu`, `nu`, `V`, `nu.e`, and `ssq`.}

\item{Mcmc}{A list with one required parameter: `R`-number of iterations, and optional parameters: `keep` and `nprint`.}

\item{verbose}{If \code{TRUE}, enumerates model parameters and timing information.}
}
\value{
A list of sharded partitions where each partition contains the following: 
\item{compdraw}{A list (length: R/keep) where each list contains `mu` (vector, length: `ncomp`) and `rooti` (matrix, shape: ncomp \eqn{\times} ncomp)}
\item{probdraw}{A \eqn{(R/keep) \times (ncomp)} matrix that reports the probability that each draw came from a particular component}
\item{Deltadraw}{A \eqn{(R/keep) \times (nz \times nvar)} matrix of draws of Delta, first row is initial value. Deltadraw is NULL if Z is NULL}
}
\description{
\code{rhierLinearMixtureParallel} implements a MCMC algorithm for hierarchical linear model with a mixture of normals heterogeneity distribution.
}
\details{
\subsection{Model and Priors}{
    \code{nreg} regression equations with \code{nvar} as the number of \eqn{X} vars in each equation \cr
    \eqn{y_i = X_i\beta_i + e_i} with \eqn{e_i} \eqn{\sim}{~} \eqn{N(0, \tau_i)}  

    \eqn{\tau_i} \eqn{\sim}{~} \eqn{nu.e*ssq_i/\chi^2_{nu.e}} where  \eqn{\tau_i} is the variance of \eqn{e_i}\cr
    \eqn{B = Z\Delta + U} or \eqn{\beta_i = \Delta' Z[i,]' + u_i} \cr
    \eqn{\Delta} is an \eqn{nz \times nvar} matrix \cr

    \eqn{Z} should \emph{not} include an intercept and should be centered for ease of interpretation. 
    The mean of each of the \code{nreg} \eqn{\beta}s is the mean of the normal mixture. 
    Use \code{summary()} to compute this mean from the \code{compdraw} output.

    \eqn{u_i} \eqn{\sim}{~} \eqn{N(\mu_{ind}, \Sigma_{ind})}\cr
    \eqn{ind} \eqn{\sim}{~} \eqn{multinomial(pvec)} \cr

    \eqn{pvec} \eqn{\sim}{~} \eqn{dirichlet(a)}\cr
    \eqn{delta = vec(\Delta)} \eqn{\sim}{~} \eqn{N(deltabar, A_d^{-1})}\cr
    \eqn{\mu_j} \eqn{\sim}{~} \eqn{N(mubar, \Sigma_j(x) Amu^{-1})}\cr
    \eqn{\Sigma_j} \eqn{\sim}{~} \eqn{IW(nu, V)} \cr
   
    Be careful in assessing the prior parameter \code{Amu}: 0.01 can be too small for some applications. 
    See chapter 5 of Rossi et al for full discussion.\cr    
}

\subsection{Argument Details}{
\emph{\code{Data  = list(regdata, Z)} [\code{Z} optional]}
\tabular{ll}{
    \code{regdata:        } \tab A \eqn{nreg} size list of regdata \cr
    \code{regdata[[i]]$X: } \tab \eqn{n_i \times nvar} design matrix for equation \eqn{i}  \cr
    \code{regdata[[i]]$y: } \tab \eqn{n_i \times 1} vector of observations for equation \eqn{i} \cr
    \code{Z:              } \tab An \eqn{(nreg) \times nz} matrix of unit characteristics 
    }
 
\emph{\code{Prior = list(deltabar, Ad, mubar, Amu, nu.e, V, ssq, ncomp)} [all but \code{ncomp} are optional]}
\tabular{ll}{
    \code{deltabar:       } \tab \eqn{(nz \times nvar) \times 1} vector of prior means (def: 0) \cr
    \code{Ad:             } \tab prior precision matrix for vec(D) (def: 0.01*I) \cr
    \code{mubar:          } \tab \eqn{nvar \times 1} prior mean vector for normal component mean (def: 0) \cr
    \code{Amu:            } \tab prior precision for normal component mean (def: 0.01) \cr
    \code{nu.e:             } \tab d.f. parameter for regression error variance prior (def: 3) \cr
    \code{V:              } \tab PDS location parameter for IW prior on normal component Sigma (def: nu*I) \cr
    \code{ssq:            } \tab scale parameter for regression error variance prior (def: \code{var(y_i)}) \cr
    \code{ncomp:          } \tab number of components used in normal mixture \cr
    }
 
\emph{\code{Mcmc  = list(R, keep, nprint)} [only \code{R} required]}
\tabular{ll}{
    \code{R:              } \tab number of MCMC draws \cr
    \code{keep:           } \tab MCMC thinning parameter -- keep every \code{keep}th draw (def: 1) \cr
    \code{nprint:         } \tab print the estimated time remaining for every \code{nprint}'th draw (def: 100, set to 0 for no print) \cr
    } 
}
}
\examples{


######### Single Component with rhierLinearMixtureParallel########
R = 500

nreg=1000
nobs=5 #number of observations
nvar=3 #columns
nz=2

Z=matrix(runif(nreg*nz),ncol=nz) 
Z=t(t(Z)-apply(Z,2,mean))

Delta=matrix(c(1,-1,2,0,1,0),ncol=nz)
tau0=.1
iota=c(rep(1,nobs)) 

#Default
tcomps=NULL
a=matrix(c(1,0,0,0.5773503,1.1547005,0,-0.4082483,0.4082483,1.2247449),ncol=3) 
tcomps[[1]]=list(mu=c(0,-1,-2),rooti=a) 
tpvec=c(1)                               

regdata=NULL						  
betas=matrix(double(nreg*nvar),ncol=nvar) 
tind=double(nreg) 

for (reg in 1:nreg) {
 tempout=bayesm::rmixture(1,tpvec,tcomps) 
 if (is.null(Z)){
   betas[reg,]= as.vector(tempout$x)  
 }else{
   betas[reg,]=Delta \%*\% Z[reg,]+as.vector(tempout$x)} 
 tind[reg]=tempout$z
 X=cbind(iota,matrix(runif(nobs*(nvar-1)),ncol=(nvar-1))) 
 tau=tau0*runif(1,min=0.5,max=1) 
 y=X \%*\% betas[reg,]+sqrt(tau)*rnorm(nobs)
 regdata[[reg]]=list(y=y,X=X,beta=betas[reg,],tau=tau)
}


Data1=list(list(regdata=regdata,Z=Z))
s = 1
Data2=scalablebayesm::partition_data(Data1,s=s)

Prior1=list(ncomp=1)
Mcmc1=list(R=R,keep=1)

out2 = parallel::mclapply(Data2, FUN = rhierLinearMixtureParallel, Prior = Prior1,
Mcmc = Mcmc1, mc.cores = s, mc.set.seed = FALSE)

######### Multiple Components with rhierLinearMixtureParallel########
R = 500

set.seed(66)
nreg=1000
nobs=5 #number of observations
nvar=3 #columns
nz=2

Z=matrix(runif(nreg*nz),ncol=nz) 
Z=t(t(Z)-apply(Z,2,mean))

Delta=matrix(c(1,-1,2,0,1,0),ncol=nz)
tau0=.1
iota=c(rep(1,nobs)) 

#Default
tcomps=NULL
a=matrix(c(1,0,0,0.5773503,1.1547005,0,-0.4082483,0.4082483,1.2247449),ncol=3) 
tcomps[[1]]=list(mu=c(0,-1,-2),rooti=a) 
tcomps[[2]]=list(mu=c(0,-1,-2)*2,rooti=a)
tcomps[[3]]=list(mu=c(0,-1,-2)*4,rooti=a)
tpvec=c(.4,.2,.4)                                   

regdata=NULL						  
betas=matrix(double(nreg*nvar),ncol=nvar) 
tind=double(nreg) 

for (reg in 1:nreg) {
 tempout=bayesm::rmixture(1,tpvec,tcomps) 
 if (is.null(Z)){
   betas[reg,]= as.vector(tempout$x)  
 }else{
   betas[reg,]=Delta \%*\% Z[reg,]+as.vector(tempout$x)} 
 tind[reg]=tempout$z
 X=cbind(iota,matrix(runif(nobs*(nvar-1)),ncol=(nvar-1))) 
 tau=tau0*runif(1,min=0.5,max=1) 
 y=X \%*\% betas[reg,]+sqrt(tau)*rnorm(nobs)
 regdata[[reg]]=list(y=y,X=X,beta=betas[reg,],tau=tau)
}


Data1=list(list(regdata=regdata,Z=Z))
s = 1
Data2=scalablebayesm::partition_data(Data1, s=s)

Prior1=list(ncomp=3)
Mcmc1=list(R=R,keep=1)

set.seed(1)
out2 = parallel::mclapply(Data2, FUN = rhierLinearMixtureParallel, Prior = Prior1,
Mcmc = Mcmc1, mc.cores = s, mc.set.seed = FALSE)

}
\references{
Bumbaca, Federico (Rico), Sanjog Misra, and Peter E. Rossi (2020), "Scalable Target Marketing:
Distributed Markov Chain Monte Carlo for Bayesian Hierarchical Models", Journal of Marketing
Research, 57(6), 999-1018.

Chapter 5, \emph{Bayesian Statistics and Marketing} by Rossi, Allenby, and McCulloch.
}
\seealso{
\code{\link{partition_data}}, 
\code{\link{drawPosteriorParallel}}, 
\code{\link{combine_draws}}, 
\code{\link{rheteroLinearIndepMetrop}}
}
\author{
Federico Bumbaca, Leeds School of Business, University of Colorado Boulder, \email{federico.bumbaca@colorado.edu}
}
