#' Computes crude/raw death rates/probabilities using triple rolling windows
#'
#' @description This function estimates (crude/raw) death rates/probabilities (life tables) using
#'              rolling windows in three dimensions (e.g., age, time and income) from a
#'              given experience of mortality, summarized in a three-dimension array of (either initial
#'              or times of) exposed to risk and a three-dimension dataset of recorded deaths.
#'
#' @author Jose M. Pavia \email{pavia@@uv.es}
#' @author Josep Lledo \email{josep.lledo@@uv.es}
#'
#' @param exposed A three-dimensional array of order n×T×I. It is recommended (and when `initial = TRUE` assumed) that the first dimension corresponds to age. For each row-column-slice combination, the array contains either the initial number of individuals exposed to risk in the portfolio (population) or the exposure time for those at risk within the portfolio. By default, the array contains exposure times. When `exposed` contains the initial number of individuals exposed to risk, (i) the first dimension represents must represent ages, and (ii) exposure times are estimated based on the assumptions of uniformity and a closed demographic system. It is also assumed that the order of rows, columns, and slices in `deaths` and `exposed` is consistent (i.e., identical) and that the sequence of values (levels) across the three dimensions is ordered sequentially.
#' @param deaths A three-dimension array of order nxTxI. It is recommended (and when `initial = TRUE` assumed) that the first dimension corresponds to age. For each combination of row-column-slice, the array contains the number of deaths recorded in the portfolio (population). It is assumed that the order of rows, columns, and slices in `deaths` and `exposed` is consistent (i.e., identical) and that the sequence of values (levels) across the three dimensions is ordered sequentially.
#' @param dim1.window An non-negative integer indicating the number of preceding and succeeding levels to be included in the rolling window for dimension 1. Default, 2.
#' @param dim2.window An non-negative integer indicating the number of preceding and succeeding levels to be included in the rolling window for dimension 2. Default, 2.
#' @param dim3.window An non-negative integer indicating the number of preceding and succeeding levels to be included in the rolling window for  dimension 3. Default, 2.
#' @param dim1.wb An non-negative integer indicating the number of preceding levels to be included in the rolling window for dimension 1. Default, `dim1.window`.  If `dim1.wb` and `dim1.window` differ, the definition of `dim1.wb` takes precedence, allowing non-centered dimension 1 windows.
#' @param dim1.wf An non-negative integer indicating the number of succeeding levels to be included in the rolling window for dimension 1. Default, `dim1.window`. If `dim1.wf` and `dim1.window` differ, the definition of `dim1.wf` takes precedence, allowing non-centered dimension 1 windows.
#' @param dim2.wb An non-negative integer indicating the number of preceding levels to be included in the rolling window for dimension 2. Default, `dim2.window`.  If `dim2.wb` and `dim2.window` differ, the definition of `dim2.wb` takes precedence, allowing non-centered dimension 2 windows.
#' @param dim2.wf An non-negative integer indicating the number of succeeding levels to be included in the rolling window for dimension 2. Default, `dim2.window`. If `dim2.wf` and `dim2.window` differ, the definition of `dim2.wf` takes precedence, allowing non-centered dimension 2 windows.
#' @param dim3.wb An non-negative integer indicating the number of preceding levels to be included in the rolling window for dimension 3. Default, `dim3.window`.  If `dim3.wb` and `dim3.window` differ, the definition of `dim3.wb` takes precedence, allowing non-centered dimension 3 windows.
#' @param dim3.wf An non-negative integer indicating the number of succeeding levels to be included in the rolling window for dimension 3. Default, `dim3.window`. If `dim3.wf` and `dim3.window` differ, the definition of `dim3.wf` takes precedence, allowing non-centered dimension 3 windows.
#' @param weights A three-dimension array of order (dim1.wb + dim1.wf + 1) x (dim2.wb + dim2.wf + 1) x (dim3.wb + dim3.wf + 1) of real positive numbers indicating the multiplicative factors to be applied to the different components of the triple rolling window. Default, `1`; meaning all the components of the rolling window are aggregated as available in `exposed` and `deaths`.
#' @param initial A TRUE/FALSE argument indicating whether the numbers in `exposed` represent the initial number of individuals exposed to risk in the portfolio (population), which corresponds to `initial = TRUE`, or the time exposed to risk, with `initial = FALSE` as default.
#' @param partial A TRUE/FALSE argument indicating whether estimates obtained using incomplete rolling windows should be also computed. Default, `FALSE`; meaning crude death rates are estimated only for combinations where the corresponding triple rolling window is complete.
#'
#' @return
##' A list with six three-dimension arrays: `mx`, `qx`, `Lx`, `dx`, `Lx.total`, `dx.total`. When `partial = FALSE` these arrays are of order (n - dim1.wb - dim1.wf) x (T - dim2.wb - dim2.wf) x (I - dim3.wb - dim3.wf); otherwise, they are of order nxTxI. The names of the rows, columns and slices of these arrays are inherited from the corresponding names in the `deaths` object, ensuring that if the row, column and slice names indicate relevant data, this contextual information is preserved.
#'  \item{mx}{ The `mx` array holds the crude death rates estimated after determining the number of individuals (time) exposed to risk and deaths to be used for each estimate using the specified rolling windows with the defined `weights`.}
#'  \item{qx}{ The `qx` array contains the raw death probabilities, derived from `mx`, based on the assumption that, on average, each deceased individual lives for half a year in the year of their death. }
#'  \item{Lx}{ The `Lx` array contains the actual time exposures used to compute the estimates. }
#'  \item{dx}{ The `dx` array contains the number of deaths used to compute the estimates.. }
#'  \item{Lx.total}{ The `Lx.total` array contains the total numbers for time exposures which corresponds to all factor-weights being equal. If `weights` is a constant matrix `Lx` and `Lx.total` coincide. }
#'  \item{dx.total}{ The `dx.total` array contains the total numbers deaths which corresponds to all factor-weights being equal. If `weights` is a constant array `dx` and `dx.total` coincide. }
#'
#' @export
#'
#' @note The first dimension in the objects `exposed` and `deaths` should correspond to the variable age when the object `exposed` represents the initial number of individuals exposed to risk in the portfolio (population).
#' The function could be used to apply triple rolling windows to datasets from other fields other than demography and actuarial science.
#'
#' @examples
#'
#' exposed0 <- structure(list(Year2017 = c(6078.14, 5841.78, 5575.70, 5726.18, 5458.21, 5197.56,
#'                                      5018.12, 4791.56, 4245.15, 4321.65, 4179.3),
#'                            Year2018 = c(5978.73, 5473.78, 5572.23, 5495.19, 5148.47, 4845.14,
#'                                      4739.54, 4222.01, 4476.99, 4306.45, 4108.58),
#'                            Year2019 = c(5593.23, 5551.41, 5260.44, 5079.56, 4873.37, 4857.78,
#'                                      4536.12, 4453.85, 4310.89, 4015.02, 3974.25)),
#'                             class = "data.frame", row.names = 68:78)
#'
#'
#' deaths0 <- structure(list(Year2017 = c(144, 102, 113, 122, 156, 110, 126, 132, 120, 172, 110),
#'                            Year2018 = c(111, 122, 109, 116, 162, 154, 115, 146, 100, 169, 146),
#'                            Year2019 = c(100, 123, 113, 151, 122, 110, 137, 175, 137, 110, 155)),
#'                           class = "data.frame", row.names = 68:78)
#'
#' exposed <- deaths <- array(NA, dim = c(11, 3, 5))
#' set.seed(123)
#' for (kk in 1:5){
#'    exposed[, , kk] <- as.matrix(exposed0) +
#'                        matrix(runif(33, 0, 100), nrow = 11, ncol = 3)
#'    deaths[, , kk] <- as.matrix(deaths0) +
#'                        matrix(runif(33, 0, 5), nrow = 11, ncol = 3)
#' }
#' dimnames(exposed) <- dimnames(deaths) <- list(68:78,
#'                                               paste0("Y", 2017:2019),
#'                                               paste0("I", 1:5))
#'
#' example <- tw_crude_mx(exposed = exposed, deaths = deaths, dim1.window = 2,
#'                        dim2.window = 1, dim3.window = 1, initial = FALSE)
#'

tw_crude_mx <- function(exposed,
                        deaths,
                        dim1.window = 2,
                        dim2.window = 2,
                        dim3.window = 2,
                        dim1.wb = dim1.window,
                        dim1.wf = dim1.window,
                        dim2.wb = dim2.window,
                        dim2.wf = dim2.window,
                        dim3.wb = dim3.window,
                        dim3.wf = dim3.window,
                        weights = 1,
                        initial = FALSE,
                        partial = FALSE){

  if (length(weights) == 1 & weights == 1)
    weights <- array(1L, dim = c(dim1.wb + dim1.wf + 1L, dim2.wb + dim2.wf + 1L,
                                 dim3.wb + dim3.wf + 1L))

  # Test inputs
  args <- as.list(environment())
  test_tw(args)

  # Initial
  if (initial){
    exposed1 <- exposed/2
    tamanyo <- dim(exposed)
    tamanyo[1L] <- tamanyo[1L] + 1L
    exposed2 <- array(0, dim = tamanyo)
    exposed2[-1, , ] <- exposed1
    exposed2 <- exposed2[-(nrow(exposed) + 1L), , ]
    exposed <- exposed1 + exposed2 - 0.5*deaths
  }

  # Expanded arrays
  tamanyo0 <- dim(deaths)
  tamanyo <- tamanyo0 + c(dim1.wb + dim1.wf, dim2.wb + dim2.wf, dim3.wb + dim3.wf)
  exposed.a <- deaths.a <- array(0, dim = tamanyo)
  exposed.a[(dim1.wb + 1L):(tamanyo[1L] - dim1.wf),
            (dim2.wb + 1L):(tamanyo[2L] - dim2.wf),
            (dim3.wb + 1L):(tamanyo[3L] - dim3.wf)] <- exposed
  deaths.a[(dim1.wb + 1L):(tamanyo[1L] - dim1.wf),
           (dim2.wb + 1L):(tamanyo[2L] - dim2.wf),
           (dim3.wb + 1L):(tamanyo[3L] - dim3.wf)] <- deaths

  # Windows aggregation of exposed and deaths
   if (max(dim1.wb, dim1.wf) == 0){
      weights <- weights[1L, , ]
   } else if (max(dim2.wb, dim2.wf) == 0) {
      weights <- weights[, 1L, ]
   } else if (max(dim3.wb, dim3.wf) == 0){
      weights <- weights[, , 1L]
   }

  Lx <- dx <- Lx.total <- dx.total <- deaths
  weights.total <- weights
  weights.total[] <- 1

  for (ii in 1L:tamanyo0[1L]){
    for (jj in 1L:tamanyo0[2L]){
      for (kk in 1L:tamanyo0[3L]){
      temp.e <- exposed.a[ii:(ii + dim1.wb + dim1.wf),
                          jj:(jj + dim2.wb + dim2.wf),
                          kk:(kk + dim3.wb + dim3.wf)]
      temp.d <- deaths.a[ii:(ii + dim1.wb + dim1.wf),
                         jj:(jj + dim2.wb + dim2.wf),
                         kk:(kk + dim3.wb + dim3.wf)]
      Lx[ii, jj, kk] <- sum(temp.e * weights)
      Lx.total[ii, jj, kk] <- sum(temp.e * weights.total)
      dx[ii, jj, kk] <- sum(temp.d * weights)
      dx.total[ii, jj, kk] <- sum(temp.d * weights.total)
      }
    }
  }
  mx <- dx/Lx
  qx <- mx/(1 + 0.5*mx)
  if(!partial){
    mx <- mx[(1L + dim1.wb):(tamanyo0[1L] - dim1.wf),
             (1L + dim2.wb):(tamanyo0[2L] - dim2.wf),
             (1L + dim3.wb):(tamanyo0[3L] - dim3.wf)]
    qx <- qx[(1L + dim1.wb):(tamanyo0[1L] - dim1.wf),
             (1L + dim2.wb):(tamanyo0[2L] - dim2.wf),
             (1L + dim3.wb):(tamanyo0[3L] - dim3.wf)]
    Lx <- Lx[(1L + dim1.wb):(tamanyo0[1L] - dim1.wf),
             (1L + dim2.wb):(tamanyo0[2L] - dim2.wf),
             (1L + dim3.wb):(tamanyo0[3L] - dim3.wf)]
    dx <- dx[(1L + dim1.wb):(tamanyo0[1L] - dim1.wf),
             (1L + dim2.wb):(tamanyo0[2L] - dim2.wf),
             (1L + dim3.wb):(tamanyo0[3L] - dim3.wf)]
    Lx.total <- Lx.total[(1L + dim1.wb):(tamanyo0[1L] - dim1.wf),
                         (1L + dim2.wb):(tamanyo0[2L] - dim2.wf),
                         (1L + dim3.wb):(tamanyo0[3L] - dim3.wf)]
    dx.total <- dx.total[(1L + dim1.wb):(tamanyo0[1L] - dim1.wf),
                         (1L + dim2.wb):(tamanyo0[2L] - dim2.wf),
                         (1L + dim3.wb):(tamanyo0[3L] - dim3.wf)]
  }
  output <- list("mx" = mx, "qx" = qx, "Lx" = Lx, "dx" = dx,
                 "Lx.total" = Lx.total, "dx.total" = dx.total)
  class(output) <- c("rwlifetable", "tw_crude_mx", "list")
  return(output)
}




test_tw <- function(args){
  if(!(is.array(args$exposed) & is.array(args$deaths)))
    stop("The objects 'exposed' and 'deaths' should be arrays of three dimensions.")
  if(!(length(dim(args$exposed)) == 3L & length(dim(args$deaths)) == 3L))
    stop("The objects 'exposed' and 'deaths' should be arrays of three dimensions.")
  if (!all(dim(args$exposed) == dim(args$deaths)))
    stop("The number of rows, columns and/or slices in 'exposed' and 'deaths' differ.")

  if (min(args$exposed) < 0 | min(args$deaths) < 0)
    stop("Negative numbers are not allowed in 'exposed' or 'deaths'.")

  if ((args$dim1.window - floor(args$dim1.window)) > 0 | args$dim1.window < 0)
    stop("The argument 'dim1.window' must be a non-negative integer.")

  if ((args$dim2.window - floor(args$dim2.window)) > 0 | args$dim2.window < 0)
    stop("The argument 'dim2.window' must be a non-negative integer.")

  if ((args$dim3.window - floor(args$dim3.window)) > 0 | args$dim3.window < 0)
    stop("The argument 'dim3.window' must be a non-negative integer.")

  if ((args$dim1.wb - floor(args$dim1.wb)) > 0 | args$dim1.wb < 0)
    stop("The argument 'dim1.wb' must be a non-negative integer.")

  if ((args$dim1.wf - floor(args$dim1.wf)) > 0 | args$dim1.wf < 0)
    stop("The argument 'dim1.wf' must be a non-negative integer.")

  if ((args$dim2.wb - floor(args$dim2.wb)) > 0 | args$dim2.wb < 0)
    stop("The argument 'dim2.wb' must be a non-negative integer.")

  if ((args$dim2.wf - floor(args$dim2.wf)) > 0 | args$dim2.wf < 0)
    stop("The argument 'dim2.wf' must be a non-negative integer.")

  if ((args$dim3.wb - floor(args$dim3.wb)) > 0 | args$dim3.wb < 0)
    stop("The argument 'dim3.wb' must be a non-negative integer.")

  if ((args$dim3.wf - floor(args$dim3.wf)) > 0 | args$dim3.wf < 0)
    stop("The argument 'dim3.wf' must be a non-negative integer.")

  if (!all(dim(args$weights) == c(args$dim1.wb + args$dim1.wf + 1L, args$dim2.wb + args$dim2.wf + 1L, args$dim3.wb + args$dim3.wf + 1L)))
    stop("The argument 'weights' must be 1 or an array of order (dim1.wb + dim1.wf + 1) x (dim2.wb + dim2.wf + 1) x (dim3.wb + dim3.wf + 1).")

  if (min(args$weights) < 0)
    stop("The values in 'weights' cannot be negative.")

  if (!args$partial & (args$dim1.wb + args$dim1.wf + 1L) > dim(args$deaths)[1L])
    stop("The dim1 rolling window is too large considering the number of levels in dimension 1 in the datasets.")

  if (!args$partial & (args$dim2.wb + args$dim2.wf + 1L) > dim(args$deaths)[2L])
    stop("The dim2 rolling window is too large considering the number of levels in dimension 2 in the datasets.")

  if (!args$partial & (args$dim3.wb + args$dim3.wf + 1L) > dim(args$deaths)[3L])
    stop("The dim3 rolling window is too large considering the number of levels in dimension 3 in the datasets.")
}
