% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulation.R
\name{populate.snp}
\alias{populate.snp}
\title{Populate SNPs.}
\usage{
populate.snp(inheritance, haplotype, marker, member.index = NULL,
  output.allele = TRUE, output.haplotype = FALSE)
}
\arguments{
\item{inheritance}{list of numeric matrices.}

\item{haplotype}{numeric matrix.}

\item{marker}{numeric vector.}

\item{member.index}{integer vector.}

\item{output.allele}{logical.}

\item{output.haplotype}{logical.}
}
\value{
A matrix of genotypic/haplotypic data. The matrix is in individual major, where marker data for each individual/meiosis are found on the same row. Exact format of the matrix depends on various input arguments.
}
\description{
\code{populate.snp} assigns alleles to markers, given inheritance information and founder haplotypes.
}
\details{
\code{inheritance} is a list of matrices produced by, e.g., \code{\link{sim.recomb}}. Each matrix contains a column of founder genome labels and a column of recombination breakpoints for the corresponding meiosis.

\code{haplotype} is a numeric matrix. The matrix is number of haplotypes by number of markers in dimension. Standard coding in this package is 1 for reference allele and 2 for alternate allele. This coding is required when \code{output.allele = FALSE}. Input data with different coding of alleles can be recoded using \code{\link{recode.snpdata}}. Number of haplotypes cannot be fewer than the number of founder genome labels in \code{inheritance}. The haplotypes will be assigned to each founder genome label in given order.

\code{marker} is a vector of marker genetic positions in Haldane centiMorgan in ascending order. Range of marker positions cannot exceed range covered by \code{inheritance}.

\code{member.index} contains indices of members in the pedigree that we wish to output data. Default value is \code{FALSE}, in which case marker data on everyone will be produced. \code{\link{get.pedindex}} can help find indices given member ID.

\code{output.allele} determines if one or two numbers will be used to represent data at each marker. Default is \code{TRUE}, in which case marker data is represented by two ordered (paternal first) alleles. Otherwise marker data is represented by a single number (0, 1 or 2) of reference alleles.

\code{output.haplotype} determines if haplotype data are separate in output. It is only used when \code{output.allele = TRUE}. Default value is \code{FALSE}, in which case each row in the output matrix represents ordered genotypes from all markers of the same individual. Otherwise each row in the output matrix represents a parental haplotype.
}
\examples{
# a simple pedigree with sibling marriage
pedigree = as.character(rep(1, 5))
member = as.character(c(11, 12, 21, 22, 31))
sex = as.numeric(c(1, 2, 1, 2, 1))
father = as.character(c(NA, NA, 11, 11, 21))
mother = as.character(c(NA, NA, 12, 12, 22))
pedinfo = data.frame(pedigree, member, sex, father, mother, stringsAsFactors = FALSE)

L = 100.0 # segment length
nsnp = 10 # number of SNPs
nhaplo = 4 # number of founder haplotypes
inher = sim.recomb(pedinfo, L)
haplo = matrix(c(3,4,4,4), nhaplo, nsnp)
marker = sort(runif(nsnp, 0, L))

# output genotype data for the 4th and 5th member
# of pedigree, genotype data displayed as two alleles
populate.snp(inher, haplo, marker, c(4, 5))

# output haplotype data for the 4th and 5th member of pedigree
populate.snp(inher, haplo, marker,c(4, 5), output.haplotype = TRUE)

# output genotype data for all members, genotype data
# displayed as counts of reference alleles
geno = recode.snpdata(haplo, input.haplotype = TRUE, output.haplotype = TRUE)[[1]]
populate.snp(inher, geno, marker, output.allele = FALSE)
}
