% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/to_flextable.R
\name{to_flextable}
\alias{to_flextable}
\alias{to_flextable.test_mediation}
\alias{to_flextable.summary_test_mediation}
\alias{to_flextable.list}
\title{Tabular summary of results from (robust) mediation analysis}
\usage{
to_flextable(object, ...)

\method{to_flextable}{test_mediation}(object, type = c("boot", "data"), ...)

\method{to_flextable}{summary_test_mediation}(object, p_value = TRUE, ...)

\method{to_flextable}{list}(
  object,
  type = c("boot", "data"),
  p_value = TRUE,
  orientation = c("landscape", "portrait"),
  ...
)
}
\arguments{
\item{object}{an object inheriting from class
\code{"\link[robmed]{test_mediation}"} or
\code{"\link[robmed:summary.test_mediation]{summary_test_mediation}"}
containing results from (robust) mediation analysis, or a list of such
objects (typically obtained via different procedures for mediation
analysis).  In case of a named list, the supplied names are used as labels
in the resulting \code{flextable}, otherwise default labels are constructed
based on how the mediation model was fitted and which type of test was used
(e.g., \code{"ROBMED"} or \code{"OLS Bootstrap"}).}

\item{\dots}{additional arguments to be passed down, eventually to
\code{\link[base]{formatC}()} for formatting numbers.  In particular,
argument \code{digits} can be used to customize the number of digits after
the decimal point (defaults to 3).  Also note that argument \code{big.mark}
is ignored for the numbers in the table; it is only used for formatting the
sample size and (if applicable) the number of bootstrap samples in the
table note.}

\item{type}{a character string specifying which estimates and significance
tests to report if mediation analysis was done via a bootstrap procedure.
If \code{"boot"} (the default), the means of the bootstrap replicates are
reported as point estimates for all effects, and significance tests for the
total and direct effects use the normal approximation of the bootstrap
distribution (i.e., the tests assume a normal distribution of the
corresponding effect with the standard deviation computed from the bootstrap
replicates).  If \code{"data"}, the point estimates on the original data are
reported for all effects, and the significance tests for the total and
direct effects are based on statistical theory (e.g., t-tests if the
coefficients are estimated via regression).  Note that for bootstrap
procedures, significance of the indirect effect is always reported via a
percentile-based confidence interval due to the asymmetry of its
distribution.}

\item{p_value}{a logical indicating whether to include p-values for the
indirect effects if mediation analysis was done via a bootstrap procedure.
If \code{TRUE} (the default), the p-values are obtained via
\code{\link[robmed]{p_value}()}.}

\item{orientation}{a character string specifying how to arrange the results
from different objects (list elements) in the \code{flextable}.  If
\code{"landscape"} (the default), results from two objects are arranged next
to each other with the results from remaining objects underneath (in groups
of two), which is intended for documents in landscape mode.  If
\code{"portrait"}, results from different objects are arranged underneath
one another, which is intended for documents in portrait mode.}
}
\value{
An object of class \code{"mediation_flextable"}, which inherits
from class \code{"\link[flextable]{flextable}"}.
}
\description{
Convert results from (robust) mediation analysis to a tabular summary in
the form of a \code{\link[flextable]{flextable}}.  This \code{flextable}
can easily be exported to \proglang{Microsoft Word} or \proglang{Microsoft
Powerpoint}, or integrated into a dynamic \proglang{Markdown} document.
This eliminates the risk of mistakes in reporting that stem from
copying-and-pasting results.
}
\note{
Numbers are not formatted via \code{flextable} defaults (see
\code{\link[flextable]{set_flextable_defaults}()}), but instead via
\code{\link[base]{formatC}()}. This is done to ensure consistency in
number formatting between functions \code{to_flextable()} and
\code{\link{to_latex}()}.
}
\examples{
data("BSG2014")

# seed to be used for the random number generator
seed <- 20150601

# perform mediation analysis via robust bootstrap test ROBMED
set.seed(seed)
robust_boot <- test_mediation(BSG2014,
                              x = "ValueDiversity",
                              y = "TeamCommitment",
                              m = "TaskConflict",
                              robust = TRUE)

# note that depending on the seed of the random number
# generator, the p value of the indirect effect may fall
# below or above the arbitrary 5\% threshold

# construct flextable of results
to_flextable(robust_boot)

# perform mediation analysis via the OLS bootstrap
set.seed(seed)
ols_boot <- test_mediation(BSG2014,
                           x = "ValueDiversity",
                           y = "TeamCommitment",
                           m = "TaskConflict",
                           robust = FALSE)

# construct flextable of results from both procedures
boot_list <- list(ols_boot, robust_boot)
to_flextable(boot_list)

# customize labels for procedures and number of digits
boot_list_named <- list("Non-robust" = ols_boot,
                        "Robust" = robust_boot)
to_flextable(boot_list_named, digits = 4)

}
\references{
Alfons, A., Ates, N.Y. and Groenen, P.J.F. (2022a) A Robust Bootstrap Test
for Mediation Analysis.  \emph{Organizational Research Methods},
\bold{25}(3), 591--617.  doi:10.1177/1094428121999096.

Alfons, A., Ates, N.Y. and Groenen, P.J.F. (2022b) Robust Mediation Analysis:
The \R Package \pkg{robmed}.  \emph{Journal of Statistical Software},
\bold{103}(13), 1--45.  doi:10.18637/jss.v103.i13.
}
\seealso{
\code{\link[robmed]{test_mediation}()},
\code{\link[robmed:summary.test_mediation]{summary}()}

\code{\link{export_docx}()}, \code{\link{export_pptx}()}

\code{\link{to_latex}()}
}
\author{
Andreas Alfons, based on code by Vincent Drenth
}
