#' Get Configuration
#'
#' Reads a YAML configuration file (default "config.yml") and returns the
#' configuration stored under the specified key (default "default"). This function
#' merges the functionality of loading the file and parsing its contents into a list.
#' It uses UTF-8 encoding to read the file and then processes it with a minimal YAML parser.
#'
#' If the parsed configuration list contains a component with the name specified by
#' `config_name`, only that component is returned. Otherwise, the entire configuration
#' list is returned.
#'
#' @param file Path to the YAML configuration file. For example, to load the sample
#'   configuration provided in the package extdata directory, use:
#'   `system.file("extdata", "config.yml", package = "rconf")`.
#' @param config_name The key in the configuration to extract (e.g., "default"). If the
#'   key exists in the parsed configuration list, only that sub-list is returned.
#' @return A list of configuration settings. When the configuration file contains multiple
#'   sections, and the section matching `config_name` is found, only that section is returned.
#'   Otherwise, the full configuration list is returned.
#'
#' @examples
#' # Example: Load sample configuration from the extdata directory.
#' cfg <- get_config(system.file("extdata", "config.yml", package = "rconf"),
#'                   config_name = "default")
#' # Access specific configuration settings:
#' print(cfg$raw_data_dir)
#' print(cfg$processed_data_dir)
#'
#' @export
#' @author
#' Yaoxiang Li
get_config <- function(file = "config.yml", config_name = "default") {
  # Read the configuration file using UTF-8 encoding.
  content <- read_utf8(file)
  # Parse the YAML content into an R list.
  conf <- parse_config(content)
  # If a specific configuration section is present, return it; otherwise, return the full configuration.
  if (!is.null(conf[[config_name]])) {
    return(conf[[config_name]])
  }
  conf
}
