% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RapData.R
\name{RapData}
\alias{RapData}
\title{Create new RapData object}
\usage{
RapData(
  pu,
  species,
  targets,
  pu.species.probabilities,
  attribute.spaces,
  boundary,
  polygons = NA,
  skipchecks = FALSE,
  .cache = new.env()
)
}
\arguments{
\item{pu}{\code{\link[base:data.frame]{base::data.frame()}} planning unit data. Columns must be
"cost" (\code{numeric}), "area" (\code{numeric}), and "status"
(\code{integer}).}

\item{species}{\code{\link[base:data.frame]{base::data.frame()}} with species data. Columns
must be "name" (\code{character}).}

\item{targets}{\code{\link[base:data.frame]{base::data.frame()}} with species data.
Columns must be "species" (\code{integer}), "target" (\code{integer}),
"proportion" (\code{numeric}).}

\item{pu.species.probabilities}{\code{\link[base:data.frame]{base::data.frame()}} with data on
the probability of species in each planning unit. Columns must be
"species", (\code{integer}), "pu" (\code{integer}), and "value"
(\code{numeric}).}

\item{attribute.spaces}{\code{list} of \code{\link[=AttributeSpaces]{AttributeSpaces()}} objects
with the demand points and planning unit coordinates.}

\item{boundary}{\code{\link[base:data.frame]{base::data.frame()}} with data on the shared
boundary length of planning units. Columns must be "id1"
(\code{integer}), "id2" (\code{integer}), and "boundary" (\code{integer}).}

\item{polygons}{\code{\link[PBSmapping:PolySet]{PBSmapping::PolySet()}} planning unit spatial data
or \code{NULL} if data not available.}

\item{skipchecks}{\code{logical} Skip data integrity checks? May improve
speed for big data sets.}

\item{.cache}{\code{\link[base:environment]{base::environment()}} used to cache calculations.}
}
\value{
A new \code{RapData} object.
}
\description{
This function creates a "RapData" object using pre-processed data.
}
\note{
Generally, users are not encouraged to change arguments to
\code{.cache}.
}
\examples{
\dontrun{
# load data
cs_pus <- sf::read_sf(
 system.file("extdata", "cs_pus.gpkg", package = "raptr")
)
cs_spp <- terra::rast(
  system.file("extdata", "cs_spp.tif", package = "raptr")
)
cs_space <- terra::rast(
  system.file("extdata", "cs_space.tif", package = "raptr")
)

# create data for RapData object
attribute.spaces <- list(
  AttributeSpaces(name = "geographic", list(
    AttributeSpace(
      planning.unit.points = PlanningUnitPoints(
        suppressWarnings(
          sf::st_coordinates(sf::st_centroid(cs_pus[1:10, ]))
        ),
        seq_len(10)
      ),
      demand.points = make.DemandPoints(
        randomPoints(cs_spp[[1]], n = 10, prob = TRUE)
      ),
      species = 1L
    ))
  ),
  AttributeSpaces(name = "environmental", list(
    AttributeSpace(
      planning.unit.points = PlanningUnitPoints(
        as.matrix(terra::extract(
          cs_space[[1]], as(cs_pus[1:10, ], "SpatVector"),
          fun = "mean",
          ID = FALSE
        )),
        seq_len(10)
      ),
      demand.points = make.DemandPoints(
        as.matrix(terra::as.data.frame(cs_space[[1]], na.rm = TRUE))
      ),
      species = 1L
    )
 ))
)
pu.species.probabilities <- calcSpeciesAverageInPus(
  cs_pus[1:10,], cs_spp[[1]]
)
polygons <- convert2PolySet(cs_pus[1:10, ])
boundary <- calcBoundaryData(cs_pus[1:10, ])

# create RapData object
x <- RapData(
  pu = cs_pus[1:10, ], species = data.frame(name = "test"),
  target = data.frame(species = 1L, target = 0:2, proportion = 0.2),
  pu.species.probabilities = pu.species.probabilities,
  attribute.spaces = attribute.spaces,
  polygons = polygons,
  boundary = boundary
)

# print object
print(x)
}
}
\seealso{
\code{\link[PBSmapping:PolySet]{PBSmapping::PolySet()}}, \code{\link[sp:SpatialPoints]{sp::SpatialPoints()}},
\code{\link[sp:SpatialPoints]{sp::SpatialPointsDataFrame()}}, \code{\link[=make.RapData]{make.RapData()}},
\linkS4class{RapData}.
}
