#' @import utils
## quiets concerns of R CMD check re: the .'s that appear in pipelines
# if(getRversion() >= "2.15.1")  utils::globalVariables(c(".",">"))
if (getRversion() >= "2.15.1") utils::globalVariables(c(".", ":=", ".x", ">"))

###################################################################
#' Infer posterior probabilities of \eqn{H_0}/\eqn{H_1} configurations.
#'
#' For each item, estimate the posterior probability for each configuration.
#' This function use either the model accounting for the dependence structure
#'  through a Gaussian copula function (\code{copula=="gaussian"}) or
#' assuming the conditional independence (\code{copula=="indep"}).
#' Utilizes parallel computing, when available. For package documentation, see \code{\link{qch-package}}.
#'
#' @param pValMat A matrix of p-values, each column corresponding to a p-value serie.
#' @param EffectMat A matrix of estimated effects corresponding to the p-values contained in \code{pValMat}.
#' If specified, the procedure will account for the direction of the effect. (optional, default is \code{NULL})
#' @param Hconfig A list of all possible combination of \eqn{H_0} and \eqn{H_1} hypotheses generated by the [GetHconfig()] function.
#' @param copula A string specifying the form of copula to use. Possible values are "\code{indep}"and "\code{gaussian}".
#' Default is "\code{indep}" corresponding to the independent case.
#' @param threads_nb The number of threads to use. The number of thread will set to the number of cores available by default.
#' @param plotting A boolean. Should some diagnostic graphs be plotted ? Default is \code{FALSE}.
#' @param Precision The precision for EM algorithm to infer the parameters. Default is \code{1e-6}.
#' @import stats
#' @importFrom qvalue pi0est
#'
#' @return A list with the following elements:
#' \tabular{ll}{
#' \code{prior} \tab vector of estimated prior probabilities
#' for each of the H-configurations.\cr
#' \code{Rcopula} \tab the estimated correlation matrix of the Gaussian copula.
#' (if applicable)\cr
#' \code{Hconfig} \tab the list of all configurations.\cr
#' \code{null_prop} \tab the estimation of items under the null for each test series.
#' }
#'
#' \itemize{
#' \item If the storage permits, the list will additionally contain:
#' \tabular{ll}{
#' \code{posterior} \tab matrix providing for each item (in row) its posterior probability to
#'  belong to each of the H-configurations (in columns).\cr
#' \code{fHconfig} \tab matrix containing \eqn{\psi_c} densities evaluated at each items,
#'  each column corresponding to a configuration.
#' }
#'
#' \item Else, the list will additionally contain:
#' \tabular{ll}{
#' \code{f0Mat} \tab matrix containing the evaluation of the marginal densities under \eqn{H_0} at each items,
#'  each column corresponding to a p-value serie.\cr
#' \code{f1Mat} \tab matrix containing the evaluation of the marginal densities under \eqn{H_1} at each items,
#'  each column corresponding to a p-value serie.\cr
#' \code{F0Mat} \tab matrix containing the evaluation of the marginal cdf under \eqn{H_0} at each items,
#'  each column corresponding to a p-value serie.\cr
#' \code{F1Mat} \tab matrix containing the evaluation of the marginal cdf under \eqn{H_1} at each items,
#'  each column corresponding to a p-value serie.\cr
#' \code{fHconfig_sum} \tab vector containing \eqn{(\sum_cw_c\psi_c(Z_i))} for each items \eqn{i}.\cr
#'  }
#'  }
#'
#' The elements of interest are the posterior probabilities matrix, \code{posterior},
#'  the estimated proportion of observations belonging to each configuration, \code{prior}, and
#'  the estimated correlation matrix of the Gaussian copula, \code{Rcopula}.
#' The remaining elements are returned primarily for use by other functions.
#'
#'
#' @export
#'
#' @examples
#' data(PvalSets_cor)
#' PvalMat <- as.matrix(PvalSets_cor[, -3])
#' ## Build the Hconfig objects
#' Q <- 2
#' Hconfig <- GetHconfig(Q)
#'
#' ## Run the function
#' res.fit <- qch.fit(pValMat = PvalMat, Hconfig = Hconfig, copula = "gaussian")
#'
#' ## Display the prior of each class of items
#' res.fit$prior
#'
#' ## Display the correlation estimate of the gaussian copula
#' res.fit$Rcopula
#'
#' ## Display the first posteriors
#' head(res.fit$posterior)
qch.fit <- function(pValMat, EffectMat = NULL, Hconfig, copula = "indep", threads_nb = 0, plotting = FALSE, Precision = 1e-6) {
  n <- nrow(pValMat)
  Q <- ncol(pValMat)

  # for numerical precision problems
  pValMat <- pValMat %>% as.matrix()

  pValMat[which(pValMat == 0)] <- min(pValMat[which(pValMat != 0)])
  pValMat[which(pValMat == 1)] <- max(pValMat[which(pValMat != 1)])

  ### Check the dimension


  ### Check parameters
  if (copula != "indep" & copula != "gaussian") {
    stop("Error. Copula must be 'indep' or 'gaussian'.")
  }
  if (copula == "gaussian" & !is.null(EffectMat)) {
    stop("Error. The procedure cannot take account of the direction of effects with a copula other than indep.")
  }

  ## Memory management
  if (n * (2**Q) * 8 > 2e9) { # if the storage requirement exceeds 2Go
    memory_management <- TRUE
  } else {
    memory_management <- FALSE
  }


  #### Zscores computation
  if (is.null(EffectMat)) {
    XMat <- -qnorm(pValMat)
  } else {
    XMat <- -qnorm(pValMat / 2) * sign(EffectMat)
  }

  #### Step 1: Marginal density estimation

  ## Get p0 estimates
  p0 <- rep(0, Q)
  for (q in 1:Q) {
    p0[q] <- min(qvalue::pi0est(p = pValMat[, q], pi0.method = "bootstrap")$pi0, 1 - 1 / n)
  }
  SomeH1 <- which(p0 < 1)
  NoH1 <- which(p0 == 1 - 1 / n)
  if (length(NoH1) == 1) {
    message(paste("Pvalue serie", NoH1, "may have very few H1 (or a weird distribution)"))
  }
  if (length(NoH1) > 1) {
    message(paste("Pvalue series", paste(NoH1, collapse = " "), "may have very few H1 (or a weird distribution)"))
  }

  ## Fit a 2-component mixture to each test serie using kerFdr
  f1Mat <- matrix(1, n, Q)
  F1Mat <- matrix(1, n, Q)
  if (is.null(EffectMat)) {
    for (q in SomeH1) {
      ker <- FastKerFdr_unsigned(XMat[, q], p0 = p0[q], plotting)
      f1Mat[, q] <- ker$f1
      F1Mat[, q] <- ker$F1
      p0[q] <- ker$p0
    }
  } else {
    for (q in SomeH1) {
      ker <- FastKerFdr_signed(XMat[, q], p0 = p0[q], plotting = FALSE)
      f1Mat[, q] <- ker$f1
      F1Mat[, q] <- ker$F1
      p0[q] <- ker$p0
    }
    f1_signed <- f1_separation_signed(XMat, f0Mat, f1Mat, p0, plotting)
  }

  f0Mat <- matrix(dnorm(XMat), ncol = Q)
  F0Mat <- pnorm(XMat, mean = 0, sd = 1)

  # for numerical precision problems
  F0Mat[which(F0Mat > 1 - 1e-12)] <- 1 - 1e-12
  F0Mat[which(F0Mat < 1e-12)] <- 1e-12
  F1Mat[which(F1Mat > 1 - 1e-12)] <- 1 - 1e-12
  F1Mat[which(F1Mat < 1e-12)] <- 1e-12


  #### Step 1bis: transform marginal densities into config densities

  Logf0Mat <- log(f0Mat)
  if (is.null(EffectMat)) {
    Logf1Mat <- log(f1Mat)
    if (!memory_management) {
      fHconfig <- sapply(Hconfig, function(h) {
        f <- rep(0, nrow(Logf0Mat))
        if (length(which(h == 1)) > 0) {
          f <- f + rowSums(Logf1Mat[, which(h == 1), drop = FALSE])
        }
        if (length(which(h == 0)) > 0) {
          f <- f + rowSums(Logf0Mat[, which(h == 0), drop = FALSE])
        }
        return(exp(f))
      })
    }
  } else {
    Logf1plusMat <- log(f1_signed$f1plusMat)
    Logf1minusMat <- log(f1_signed$f1minusMat)
    if (!memory_management) {
      fHconfig <- sapply(Hconfig, function(h) {
        f <- rep(0, nrow(f0Mat))
        if (length(which(h == "+")) > 0) {
          f <- f + rowSums(Logf1plusMat[, which(h == "+"), drop = FALSE])
        }
        if (length(which(h == "-")) > 0) {
          f <- f + rowSums(Logf1minusMat[, which(h == "-"), drop = FALSE])
        }
        if (length(which(h == 0)) > 0) {
          f <- f + rowSums(Logf0Mat[, which(h == 0), drop = FALSE])
        }
        return(exp(f))
      })
    }
  }

  #### Step 2: Infer parameters using an EM procedure

  ## Initialization:
  # Weights: Simple product of marginal priors estimator
  if (is.null(EffectMat)) {
    NewPrior <- sapply(1:length(Hconfig), function(c) {
      prod(p0[which(Hconfig[[c]] == 0)]) * prod(1 - p0[which(Hconfig[[c]] == 1)])
    })
  } else {
    NewPrior <- sapply(1:length(Hconfig), function(c) {
      prod(p0[which(Hconfig[[c]] == 0)]) * prod(f1_signed$p1plus[which(Hconfig[[c]] == "+")]) * prod(f1_signed$p1minus[which(Hconfig[[c]] == "-")])
    })
  }


  # Copula parameters: Kendall's tau estimator
  if (copula == "gaussian") {
    NewRho <- copula::fitCopula(copula = copula::normalCopula(dim = Q, dispstr = "un"), data = pValMat[sample(1:n, size = min(1e4, n)), ], method = "itau") %>% coef()
    NewR <- copula::p2P(param = NewRho, d = Q)
  }

  ## EM calibration
  if (copula == "indep") {
    if (!memory_management) {
      EM.res <- EM_calibration_indep(
        fHconfig = fHconfig,
        Prior.init = NewPrior,
        Precision = Precision
      )
    } else {
      EM.res <- EM_calibration_indep_memory(
        Logf0Mat = Logf0Mat,
        Logf1Mat = Logf1Mat,
        Prior.init = NewPrior,
        Hconfig = Hconfig,
        threads_nb = threads_nb,
        Precision = Precision
      )
    }
  } else if (copula == "gaussian") {
    if (!memory_management) {
      EM.res <- EM_calibration_gaussian(
        Hconfig = Hconfig,
        F0Mat = F0Mat,
        F1Mat = F1Mat,
        fHconfig = fHconfig,
        R.init = NewR,
        Prior.init = NewPrior,
        Precision = Precision
      )
    } else {
      EM.res <- EM_calibration_gaussian_memory(
        Logf0Mat = Logf0Mat,
        Logf1Mat = Logf1Mat,
        F0Mat = F0Mat,
        F1Mat = F1Mat,
        Prior.init = NewPrior,
        R.init = NewR,
        Hconfig = Hconfig,
        threads_nb = threads_nb,
        Precision = Precision
      )
    }
  }

  #### Step 2bis: Posterior computation
  if (!memory_management) {
    if (copula == "gaussian") {
      fcopula.Hconfig <- Copula.Hconfig_gaussian_density(Hconfig, F0Mat, F1Mat, EM.res$Rcopula) * fHconfig
      posterior <- fcopula.Hconfig * (tcrossprod(rep(1:n), EM.res$priorHconfig))
      posterior <- posterior / rowSums(posterior)
    } else if (copula == "indep") {
      posterior <- fHconfig * (tcrossprod(rep(1:n), EM.res$priorHconfig))
      posterior <- posterior / rowSums(posterior)
    }
  } else {
    if (copula == "gaussian") {
      RcopulaInv <- solve(EM.res$Rcopula)
      zeta0 <- qnorm(p = F0Mat, mean = 0, sd = 1)
      zeta1 <- qnorm(p = F1Mat, mean = 0, sd = 1)
      fHconfig_sum <- fHconfig_sum_update_gaussian_copula_ptr_parallel(Hconfig,
        EM.res$priorHconfig,
        Logf0Mat, Logf1Mat,
        zeta0, zeta1,
        EM.res$Rcopula, RcopulaInv,
        threads_nb = threads_nb
      )
    } else if (copula == "indep") {
      fHconfig_sum <- fHconfig_sum_update_ptr_parallel(Hconfig,
        EM.res$priorHconfig,
        Logf0Mat, Logf1Mat,
        threads_nb = threads_nb
      )
    }
  }


  #### Last but not least: output results
  if (!memory_management) {
    Res <- list(
      prior = EM.res$priorHconfig,
      posterior = posterior,
      fHconfig = fHconfig,
      Rcopula = EM.res$Rcopula,
      Hconfig = names(Hconfig),
      null_prop = p0
    )
  } else {
    Res <- list(
      prior = EM.res$priorHconfig,
      f0Mat = f0Mat, f1Mat = f1Mat,
      F0Mat = F0Mat, F1Mat = F1Mat,
      fHconfig_sum = fHconfig_sum,
      Rcopula = EM.res$Rcopula,
      Hconfig = names(Hconfig),
      null_prop = p0
    )
  }

  return(Res)
}
