#' @name nls_results
#' @title Array of Results from all NLS Models eq1-eq11
#'
#' @description This function generates an array of results from the NLS models
#'     presented in Lobo et al. (2013) and Davis et al. (2024). The array
#'     includes all calculated values/model parameters for each NLS model (eq),
#'     predicted values of the NLS curve, and goodness of fit statistics -
#'     mean squared error (MSE) and r2 values- for each predicted curve and
#'     calculated value.
#'
#' @usage
#' nls_results(
#'     data,
#'     species_col,
#'     sample_col,
#'     A_col,
#'     specs = NULL,
#'     dat_wide = NULL,
#'     PARi = c(0, 50, 100, 250, 500, 1000, 1500, 2000, 2500),
#'     PARi_fine = seq(0,6000,by = .1),
#'     eqs = paste("eq", c(1, 2, 3, 4, 5, 6, 8, 9, 11), sep = ""),
#'     par_names = c("Pgmax", "Pmax", "phi_I0", "phi_I0_Icomp", "phi_Icomp",
#'                    "phi_Icomp_200", "Rd", "Icomp", "Isat", "Isat_25",
#'                    "Isat_50", "Isat_75", "Isat_85", "Isat_90", "Isat_95",
#'                    "I5", "I10", "I15", "I25", "I50", "I75", "I85", "I90",
#'                    "I95", "Imax", "Imax_obs", "P_Imax",
#'                    "beta", "gamma", "theta", "k"))
#'
#' @param data A data frame of measured input data with a column
#'     for each unique plant observed, a column to indicate a common identifier
#'     among each unique observation a column of light intensities at measured
#'     PARi values, and a column for measured measured carbon assimilation
#'     values.
#' @param sample_col Identify the column in your data set with the Sample ID,
#'     or the unique identifier for each observation (Ex. H.Debilis_2_Oct1).
#' @param species_col Identify the column in your data set for the Species
#'     name, or the common identifier for each sample (Ex. H.Debilis)
#' @param A_col Identify the column in your data set corresponding to the
#'     measured carbon assimilation value (A).
#' @param specs A character or factor vector to subset individual identifiers,
#'     defaults to all unique samples in the data set.
#' @param dat_wide A wide-format data frame used to extract observed `PARi` and
#'     `A` values for each `Species` and `Sample` taken, defaults to
#'     transform `dat_wide` from `data` automatically.
#' @param eqs A character vector of equation names to be fitted. Defaults to the
#'     9 equations programmed in this package:
#'     (`eq1`, `eq2`, `eq3`, `eq4`, `eq5`, `eq6`, `eq8`, `eq9`, `eq11`).
#' @param par_names A character or factor vector of parameters calculated from
#'     the NLS models to include in the array of results. Defaults to parameters
#'     calculated from all 9 models in this package unless otherwise specified.
#' @param PARi A numeric vector of `PARi` values in which carbon assimilation
#'     was measured.
#'     Defaults to c(0, 50, 100, 250, 500, 1000, 1500, 2000, 2500).
#' @param PARi_fine A finer numeric vector of `PARi` values for estimating
#'     response across a continuous light range, defaults to seq(0,6000,by = .1)
#'     for fine scale predictions.
#'
#' @return A list containing the following elements:
#' \describe{
#'   \item{res}{An array of results. The array dimensions are:
#'    number of photosynthetic models (eq) tested x
#'    (number of parameters + 2 * length(PARi)) x length(inds).}
#'   \item{fitted_curves}{A list of fitted curves for each equation and
#'    individual.}
#'   \item{r2}{A matrix of `r2` values for each individual and equation.}
#'   \item{mse}{A matrix of `mse` values for each individual and equation.}
#' }
#'
#' @details The function fits each equation from the `eqs` list to the subset
#'     of `data` corresponding to each unique sample. It then returns
#'     calculated parameters, predicted values, and observed `PARi` values for
#'     the given individual(s). The results are stored in an array, and the
#'     goodness-of-fit metrics (`r2` and `mse`) are saved in separate matrices.
#'
#' @references
#'     Davis, R.E., C. M. Mason, E. W. Goolsby 2024 Comparative evolution of
#'     photosynthetic light response curve: approaches and pitfalls in
#'     phylogenetic modeling of a function-valued trait. IJPS, in review
#'
#'     Lobo, F. de A., M. P. de Barros, H. J. Dalmagro,  .C. Dalmolin,
#'     W. E. Pereira, É.C. de Souza, G. L. Vourlitis and C. E. Rodriguez Ortiz
#'     2013 Fitting net photosynthetic light-response curves with Microsoft
#'     Excel – a critical look at the models. Photosynthetica 51 (3): 445-456.
#'
#' @examples
#' # Example dataset
#' data(sunflowers)
#' data <- sunflowers |>filter(SampleID==SampleID)
#'
#' # Specify arguments
#' PARi = c(0, 50, 100, 250, 500, 1000, 1500, 2000, 2500)
#' PARifine <- seq(0, 1500, by = 1)
#' par_names <- c("Pmax", "Icomp", "phi_Icomp", "Rd", "I15, I25", "I85", "I95")
#' dat_wide <- data %>% pivot_wider(id_cols = c(Species,SampleID),
#'                                  names_from = PARi, names_prefix = "PARi_",values_from = A)
#' subset <- first_five <- unique(data$SampleID[1:43])
#'
#' # Process light response curve data for all model equations
#' my_results <- nls_results(data = data, species_col = "Species",
#'                           sample_col = "SampleID", A_col = "A", PARi_fine = PARifine,
#'                          specs = subset, dat_wide = dat_wide, par_names = par_names)
#'
#' # Access the results array
#' result_array <- my_results$res
#' # Access the fitted curves
#' fitted_curves <- my_results$fitted_curves
#' # Access r2 and mse matrices
#' r2_matrix <- my_results$r2
#' mse_matrix <- my_results$mse
#'
#' @importFrom tidyr pivot_longer pivot_wider drop_na
#' @importFrom dplyr %>%
#' @importFrom stats lm coef nls nls.control cor optim approx setNames
#' @export
nls_results <- function(data,
                        species_col,
                        sample_col,
                        A_col,
                        specs = NULL,
                        dat_wide = NULL,
                        PARi = c(0, 50, 100, 250, 500, 1000, 1500, 2000, 2500),
                        PARi_fine = seq(0,6000,by = .1),
                        eqs = paste("eq", c(1, 2, 3, 4, 5, 6, 8, 9, 11), sep = ""),
                        par_names = c("Pgmax", "Pmax", "phi_I0", "phi_I0_Icomp",
                                      "phi_Icomp", "phi_Icomp_200", "Rd", "Icomp",
                                      "Isat", "Isat_25", "Isat_50", "Isat_75",
                                      "Isat_85", "Isat_90", "Isat_95", "I5", "I10",
                                      "I15", "I25", "I50", "I75", "I85", "I90",
                                      "I95", "Imax", "Imax_obs", "P_Imax",
                                      "beta", "gamma", "theta", "k")) {
  # Remove rows with missing values in critical columns
  data <- data |> drop_na({{species_col}}, {{sample_col}}, {{A_col}})

  # Optionally subset species from inds
  inds <- unique(data[[sample_col]])
  species_subset <- if (!is.null(specs)) specs else inds

  # Transform data to long data format
  data_wide <- if (!is.null(dat_wide)) dat_wide else {
    data |>
      drop_na({{species_col}}, {{sample_col}}, {{A_col}}) |>  # Ensure no NAs in wide data
      pivot_wider(
        id_cols = c({{species_col}}, {{sample_col}}),
        names_from = PARi,
        names_prefix = "PARi_",
        values_from = {{A_col}}
      )
  }

  # Initialize lists for fitted curves and results
  fitted_curves <- setNames(vector("list", length(eqs)), eqs)
  fitted_curves <- lapply(fitted_curves, function(X) setNames(
    vector("list", length(species_subset)),
    species_subset
  ))

  nls_fits <- curve_derived_results <- nls_fits_sp <-
    curve_derived_results_sp <- setNames(vector("list", length(eqs)), eqs)

  # Create an array for results
  res <- array(dim = c(9, length(par_names) + length(PARi) * 2, length(species_subset)))
  dimnames(res) <- list(paste("eq", c(1:6, 8:9, 11), sep = ""),
                        c(par_names, paste("PAR_", PARi, sep = ""),
                          paste("obs_PAR_", PARi, sep = "")), species_subset)

  # Initialize matrices for r2 and mse
  mse <- r2 <- matrix(NA, nrow = length(species_subset), ncol = 9,
                      dimnames = list(species_subset, eqs))

  # Loop through equations and individuals
  for (e in 1:9) {
    for (i in 1:length(species_subset)) {
      try({
        # Filter data for the current individual
        individual_data <- data[data[[sample_col]] == species_subset[i], ] |>
          drop_na()  # Ignore rows with missing values for the individual

        out <- get(eqs[e])(data = individual_data, return = "all")
        fitted_curves[[e]][[species_subset[i]]] <- out
        res[e, , i] <- c(out$calc[par_names],
                         out$pred_fine[which(PARi_fine %in% PARi)],
                         unlist(data_wide[data_wide[[sample_col]] == species_subset[i],
                                          paste("PARi_", PARi, sep = "")]))
        r2[i, e] <- out$fit[["r2"]]
        mse[i, e] <- out$fit[["mse"]]
      }, silent = TRUE)
    }
  }

  # Return the results
  return(list(res = res, fitted_curves = fitted_curves, r2 = r2, mse = mse))
}
