

#' Computes the joint probabilities of an ordinal time series
#'
#' \code{joint_probabilities} returns a matrix with the joint
#' probabilities of an ordinal time series
#'
#' @param series An OTS.
#' @param lag The considered lag (default is 1).
#' @param states A numerical vector containing the corresponding
#' states.
#' @return A matrix with the joint probabilities.
#' @examples
#' matrix_jp <- joint_probabilities(series = AustrianWages$data[[100]],
#' states = 0 : 5) # Computing the matrix of
#' # joint probabilities for one series in dataset AustrianWages
#' @details
#' Given an OTS of length \eqn{T} with range \eqn{\mathcal{S}=\{s_0, s_1, s_2, \ldots, s_n\}} (\eqn{s_0 < s_1 < s_2 < \ldots < s_n}),
#' \eqn{\overline{X}_t=\{\overline{X}_1,\ldots, \overline{X}_T\}}, the function computes the
#' matrix \eqn{\widehat{\boldsymbol P}(l) = \big(\widehat{p}_{i-1j-1}(l)\big)_{1 \le i, j \le n+1}},
#' with \eqn{\widehat{p}_{ij}(l)=\frac{N_{ij}(l)}{T-l}}, where \eqn{N_{ij}(l)} is the number
#' of pairs \eqn{(\overline{X}_t, \overline{X}_{t-l})=(s_i,s_j)} in the realization \eqn{\overline{X}_t}.
#' @encoding UTF-8
#' @author
#' Ángel López-Oriona, José A. Vilar
#' @references{
#'
#'   \insertRef{weiss2019distance}{otsfeatures}
#'
#' }
#' @export

joint_probabilities <- function(series, lag = 1, states) {

  check_ots(series)
  n_states <- length(states)
  matrix_joint_probabilities <- base::matrix(0, n_states, n_states)

  for (i in 1 : n_states) {

    for (j in 1 : n_states) {

      matrix_joint_probabilities[i, j] <- p_i_j_k_function(series = series, i_stat  = states[i],
                                                           j_stat = states[j], k = lag)

    }

  }

  return(matrix_joint_probabilities)

}
