#' Sparse Quadratic Programming Solver
#'
#' Solves \deqn{arg\min_x 0.5 x'P x + q'x}{argmin_x 0.5 x'P x + q'x}
#' s.t. \deqn{l_i < (A x)_i < u_i}{li < (A x)i < ui}
#' for real matrices P (nxn, positive semidefinite) and A (mxn) with m number of constraints
#' @param P,A sparse matrices of class dgCMatrix or coercible into such, with P positive semidefinite.
#' Only the upper triangular part of P will be used.
#' @param q,l,u Numeric vectors, with possibly infinite elements in l and u
#' @param pars list with optimization parameters, conveniently set with the function \code{osqpSettings}
#' @return A list with elements x (the primal solution), y (the dual solution), prim_inf_cert,
#' dual_inf_cert, and info.
#' @seealso \code{\link{osqp}}. The underlying OSQP documentation: \url{https://osqp.org/}
#' @examples
#' library(osqp)
#' ## example, adapted from OSQP documentation
#' library(Matrix)
#'
#' P <- Matrix(c(11., 0.,
#'               0., 0.), 2, 2, sparse = TRUE)
#' q <- c(3., 4.)
#' A <- Matrix(c(-1., 0., -1., 2., 3.,
#'               0., -1., -3., 5., 4.)
#'               , 5, 2, sparse = TRUE)
#' u <- c(0., 0., -15., 100., 80)
#' l <- rep_len(-Inf, 5)
#'
#' settings <- osqpSettings(verbose = TRUE)
#'
#' # Solve with OSQP
#' res <- solve_osqp(P, q, A, l, u, settings)
#' res$x
#'
#' @references{
#' Stellato, B., Banjac, G., Goulart, P., Bemporad, A., Boyd and S. (2018).
#' ``OSQP: An Operator Splitting Solver for Quadratic Programs.''
#' \emph{ArXiv e-prints}.
#' 1711.08013.}
#' @export
solve_osqp <- function(P=NULL, q=NULL, A=NULL, l=NULL, u=NULL, pars = osqpSettings()) {

  model = osqp(P, q, A, l, u, pars)
  model@Solve()
}
