% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/customboot.R
\name{boot.modmed.mlm.custom}
\alias{boot.modmed.mlm.custom}
\title{Bootstrapping multilevel mediation model (without boot package)}
\usage{
boot.modmed.mlm.custom(
  data,
  L2ID,
  ...,
  return.type = "all",
  modval1 = NULL,
  modval2 = NULL,
  nrep = 500,
  boot.type = c("caseboth", "case2", "case1", "resid"),
  parallel.type = c("lapply", "parallel", "furrr"),
  ncores = NULL,
  seed = NULL
)
}
\arguments{
\item{data}{Data frame in long format. The function will do restructuring using \code{\link{stack_bpg}}.}

\item{L2ID}{Name of column that contains grouping variable in 'data' (e.g., "SubjectID")}

\item{...}{Arguments passed to \code{\link{modmed.mlm}} or \code{\link[nlme]{lme}} to define the mediation analysis model or do estimation, respectively.}

\item{return.type}{Character that defines what information to extract from the model. Default and options are in \code{\link{extract.modmed.mlm}}.
As examples, "indirect" will compute the indirect effect, "all" will save all random and fixed effects for possible additional
computations, "indirect.diff" will compute the difference in the indirect effect at two values of a possible moderating variable.}

\item{modval1}{(Optional) Numeric. If the model has a moderator, this value will be passed to \code{\link{extract.modmed.mlm}}
to compute the indirect effect or other effects at that value. See \code{\link{extract.modmed.mlm}} for details.}

\item{modval2}{(Optional). If the model has a moderator, it is possible to compute the difference in the indirect
at two values of the moderator. If given and an appropriate option for such a difference is chosen for \code{type},
this value and that of \code{modval1} will be passed to \code{\link{extract.modmed.mlm}} to compute and save the difference.
This is useful for obtaining a CI for the difference in the indirect effect at two different levels of the moderator.}

\item{nrep}{Number of bootstrap replications to perform. Pick a small number just for testing purposes, something larger (e.g., 1000 or more) for analyses.}

\item{boot.type}{Character indicating the type of bootstrapping to perform. Options are: "caseboth", "case2", "case1", or "resid".}

\item{parallel.type}{Character indicating type of parallel processing (if any) to use. Options are "lapply" (no parallel processing),"parallel"
(uses \code{\link[parallel]{parallel}} package), or "furrr" (uses \code{\link[furrr]{furrr}} package.}

\item{ncores}{Integer indicating the number of processing cores to attempt to use.}

\item{seed}{Integer to set random number seed, for replication purposes. Note that replication may be somewhat R version or platform dependent.}
}
\value{
A list with the following elements. Note that \code{t0} and \code{t} are intended to trick the \code{\link[boot]{boot}}
  package into working with some if its functions.
\itemize{
 \item{\code{call} Call/arguments used when invoking this function. Useful for later extracting things like indirect effect.}
 \item{\code{t0} Parameter estimates based on the dataset.}
 \item{\code{t} Bootstrap distribution of all parameter estimates.}
 \item{\code{model} Fitted model to restructured data as one would obtain from \code{\link{modmed.mlm}}.}
 \item{\code{conv} Whether model fit to restructured dataset converged.}
 \item{\code{args} Arguments used when calling \code{\link{modmed.mlm}}. Useful for later extracting things like indirect effect.}
}
}
\description{
Bootstrapping multilevel mediation model (without boot package)
}
\details{
This function was written to do all four kinds of bootstrapping outlined in Falk, Vogel, Hammami & Miočević (in press):
 case resampling at both levels, at level 2 only, at level 1 only, and the residual-based bootstrap (e.g., see Hox and van de Schoot, 2013;
 van der Leeden, Meijer, & Busing, 2008). These functions also support moderated mediation. See also \code{\link{modmed.mlm}}.
 Note that \code{\link{nlm}} was used as the optimizer for some of the examples below as it was found to be faster for the models/simulations
 studied by Falk et al (in press). Note that Level 1 only bootstrapping is typically not recommended. See Falk et al. (in press) for details.

 This function is different from the original functions used for the publication and that as of this writing still appear here: \code{\link{boot.modmed.mlm}}
 and here: \code{\link{bootresid.modmed.mlm}} . The present function seeks to unify case bootstrapping and residual-based bootstrapping in the same function. Furthermore,
 this newer function is also aimed at attempting to bypass the need for using the \code{boot} package to do computations and parallel processing.
 Some performance gains in terms of speed have been observed via use of this function instead of \code{boot} in conjunction with \code{\link{boot.modmed.mlm}}.
 Although somewhat slower, \code{\link[furrr]{furrr}} can also be used if one would like a progress bar.
}
\examples{

\donttest{

data(BPG06dat)

# Note that for all examples below, nrep should be increased to something
#  MUCH larger (e.g., 1000). Small values here are used only so that the code
#  runs relatively quickly when tested.

# double bootstrap, no parallel processing
boot.result<-boot.modmed.mlm.custom(BPG06dat, nrep=10, L2ID="id", X="x", Y="y", M="m",
  boot.type="caseboth",
  control=list(opt="nlm"), seed=1234)

extract.boot.modmed.mlm(boot.result, type="indirect", ci.conf=.95)

# residual bootstrap, parallel package
boot.result<-boot.modmed.mlm.custom(BPG06dat, nrep=10, L2ID="id", X="x", Y="y", M="m",
  boot.type="resid", random.a=TRUE, random.b=TRUE,
  parallel.type="parallel",ncores=2,seed=2299,
  control=list(opt="nlm"))

extract.boot.modmed.mlm(boot.result, type="indirect", ci.conf=.95)
}

\donttest{
# Example with moderation
data(simdat)

# moderation
boot.result<-boot.modmed.mlm.custom(simdat, nrep=5, L2ID = "L2id", X = "X", Y = "Y", M = "M",
   boot.type="caseboth",
   random.a=TRUE, random.b=TRUE, random.cprime=TRUE,
   moderator = "mod", mod.a=TRUE, mod.b=TRUE,
   random.mod.a = TRUE, random.mod.b = TRUE,
  parallel.type="parallel",ncores=2,seed=2299)

extract.boot.modmed.mlm(boot.result, type="indirect")

# indirect effect point estimate and 95\% CI when moderator = 0
extract.boot.modmed.mlm(boot.result, type="indirect", modval1=0)

# indirect effect point estimate and 95\% CI when moderator = 1
 extract.boot.modmed.mlm(boot.result, type="indirect", modval1=1)

# indirect effect difference point estimate and 95\% CI
extract.boot.modmed.mlm(boot.result, type="indirect.diff",
  modval1=0, modval2=1)

}
}
\references{
Bauer, D. J., Preacher, K. J., & Gil, K. M. (2006). Conceptualizing and testing random indirect effects and moderated mediation in multilevel models: New procedures and recommendations. Psychological Methods, 11(2), 142–163. \doi{10.1037/1082-989X.11.2.142}

Falk, C. F., Vogel, T., Hammami, S., & Miočević, M. (in press). Multilevel mediation analysis in R: A comparison of bootstrap and Bayesian approaches. Behavior Research Methods. \doi{10.3758/s13428-023-02079-4}  Preprint: \doi{10.31234/osf.io/ync34}

Hox, J., & van de Schoot, R. (2013). Robust methods for multilevel analysis. In M. A. Scott, J. S. Simonoff & B. D. Marx (Eds.), The SAGE Handbook of Multilevel Modeling (pp. 387-402). SAGE Publications Ltd. \doi{10.4135/9781446247600.n22}

van der Leeden, R., Meijer, E., & Busing, F. M. T. A. (2008). Resampling multilevel models. In J. de Leeuw & E. Meijer (Eds.), Handbook of Multilevel Analysis (pp. 401-433). Springer.
}
