#' @title Modality Assessment in Multimodal Generalized Linear Models
#'
#' @description Provides statistical inference for modality-specific information gain in
#' multimodal GLMs. Estimates ERE and pseudo-R² with confidence intervals and p-values using
#' Sure Independence Screening for variable selection and penalized likelihood for inference.
#'
#'
#' @param X The \eqn{n \times p} data matrix consisting of features from all modalities.
#' @param y The \eqn{n \times 1} vector of response.
#' @param mod.idx A list of column indices for all modalities in the concatenated data matrix \code{X}.
#' @param family A description of the error distribution and link function to be used in the model.
#' Currently, we allow the Binomial ("binomial") and
#' Gaussian ("gaussian") families with canonical links only.
#' @param iter Specifies whether to perform iterative SIS. The default is
#' \code{iter=TRUE}.
#' @param penalty Specifies the type of penalty to be used in the variable selection and
#' inference procedure.
#' Options include \code{'MCP'}, \code{'SCAD'}, and \code{'lasso'}. The default is \code{penalty='SCAD'}.
#' @param tune Specifies the method for selecting the optimal tuning parameters in (I)SIS and
#' penalized likelihood procedure. Options include \code{'bic'}, \code{'ebic'} and \code{'aic'}.
#' The default is \code{tune='bic'}.
#' @param lambda A user-specified decreasing sequence of lambda values for penalized likelihood
#' procedure. By default, a sequence of values of length \code{nlambda} is automatically computed and
#' equally spaced on the log scale.
#' @param nlambda The number of lambda values. The default is 100.
#' @param conf.level Levels of the confidence interval. The default is \code{conf.level=0.95}.
#' @param CI.type A string specifying the type of the confidence interval. Options include
#' \code{'two.sided'} and \code{'one.sided'}. The default is \code{CI.type='two.sided'}.
#' @param trace Specifies whether to print out logs of iterations in SIS procedure. The default is
#' \code{trace=FALSE}.
#'
#' @return An object with S3 class "mglm.test" containing:
#' \item{sel.idx}{ List of indices of selected features by (I)SIS in each modality.}
#' \item{num.nonzeros}{Number of selected features by (I)SIS in each modality.}
#' \item{ERE}{Point estimation of ERE for each modality.}
#' \item{ERE.CI.L}{Lower bound of the confidence interval of ERE for each modality}
#' \item{ERE.CI.U}{Upper bound of the confidence interval of ERE for each modality}
#' \item{R2}{Point estimate of pseudo-\eqn{R^2} for each modality.}
#' \item{R2.CI.L}{Lower bound of the confidence interval of pseudo-\eqn{R^2} for each modality}
#' \item{R2.CI.U}{Upper bound of the confidence interval of pseudo-\eqn{R^2} for each modality}
#' \item{conf.level}{ Level of confidence intervals.}
#' @export
#'
#' @import tidyverse
#' @importFrom MASS mvrnorm glm.nb
#' @importFrom SIS SIS
#' @importFrom glmnet glmnet
#' @importFrom glmnet cv.glmnet
#' @importFrom ncvreg ncvreg
#' @importFrom ncvreg cv.ncvreg
#' @importFrom MBESS conf.limits.nc.chisq
#' @importFrom survival coxph
#' @importFrom dplyr case_when
#' @importFrom utils capture.output

#' @examples
#' ## Example 1: Linear model
#' data(data_linear_model)
#' X <- data_linear_model$X
#' y <- data_linear_model$y
#' mod.idx <- data_linear_model$mod.idx
#' test <- mglm.test(X = X, y = y, mod.idx = mod.idx, family = "gaussian",
#'                iter = TRUE, penalty = "SCAD", tune = "bic",
#'                conf.level = 0.95, CI.type = "one.sided")
#' summary(test)
#'
#'
#' ## Example 2: Logistic regression
#' data(data_logistic_model)
#' X <- data_logistic_model$X
#' y <- data_logistic_model$y
#' mod.idx <- data_logistic_model$mod.idx
#' test <- mglm.test(X = X, y = y, mod.idx = mod.idx, family = "binomial",
#'                iter = TRUE, penalty = "SCAD", tune = "bic",
#'                conf.level = 0.95, CI.type = "two.sided")
#' sum.test <- summary(test)
#'
#'
#'
mglm.test <- function(X, y, mod.idx, family = c("gaussian", "binomial"), iter = TRUE,
                 penalty = c("SCAD", "MCP", "lasso"),
                 tune = c("bic", "ebic", "aic"),
                 lambda = NULL, nlambda = 100,
                 conf.level = 0.95, CI.type = c("two.sided", "one.sided"),
                 trace = FALSE){

  ###########################################################################################
  # Input argument checks and modifications
  ###########################################################################################

  family = match.arg(family)
  penalty = match.arg(penalty)
  tune = match.arg(tune)
  CI.type = match.arg(CI.type)


  if (is.null(X) || is.null(y))
    stop("The data is missing!")
  if (!(family %in% c("gaussian","binomial")))
    stop("Currently the function allows the Binomial ('binomial') and Gaussian ('gaussian') families with canonical links only.")
  if (!(penalty %in% c("SCAD", "MCP", "lasso")))
    stop("Selected penalty currently not available.")
  if (!(tune %in% c("bic", "ebic", "aic")))
    stop("Selected tuning method for tuning the regularization parameter not available.")

  if (!CI.type %in% c("two.sided","one.sided"))
    stop("CI.type must be 'two.sided' or 'one.sided'.")

  nobs = nrow(X)
  nvars = ncol(X)
  nMod = length(mod.idx)
  sel.idx = list()
  ERE = c()
  num.nonzeros = c()
  ERE.CI.L = c()
  ERE.CI.U = c()


  ## Conduct (I)SIS for dimension reduction
  # var_id0 <- ISIS_mglm(X=X, y=y, family=family, iter = iter, tune = tune)
  if(trace){
    sis <- SIS(x=X, y=y, family=family, iter = iter, tune = tune)
  }else{
    invisible(capture.output(sis <- SIS(x=X, y=y, family=family, iter = iter, tune = tune)))
  }
  var_id0 <- sis$ix
  if(length(var_id0)==0){
    warning("No variables selected after SIS")
  }

  for(i in 1:nMod){
    Mbeta = setdiff(1:ncol(X),mod.idx[[i]])

    ## partially penalized regularization procedure
    pen.factor = as.numeric(var_id0 %in% Mbeta)
    if(sum(pen.factor)!=0){
      tunefit = tune.fit.adj(X[, var_id0], y, family=family , penalty = penalty, tune=tune,
                             lambda = lambda, nlambda = nlambda,
                             penalty.factor = pen.factor)
      if(length(tunefit$ix) != 0) var_id = var_id0[tunefit$ix]
    }else{
      var_id = var_id0
    }

    ## Estimation of the ERE
    fit = mod_est(X, y, var_id, Mbeta, family, conf.level = conf.level, CI.type = CI.type)
    sel.idx[[i]] <- var_id[!(var_id %in% Mbeta)]
    num.nonzeros <- c(num.nonzeros,fit$num.nonzeros)
    ERE <- c(ERE,fit$Th)
    ERE.CI.L <- c(ERE.CI.L,fit$CI[1])
    ERE.CI.U <- c(ERE.CI.U,fit$CI[2])
  }

  R2 = (1 - exp(-ERE))
  R2.CI.L = (1-exp(-ERE.CI.L))
  R2.CI.U = (1-exp(-ERE.CI.U))

  result = list(sel.idx=sel.idx, num.nonzeros=num.nonzeros,
                ERE=ERE, ERE.CI.L=ERE.CI.L, ERE.CI.U=ERE.CI.U,
                R2=R2, R2.CI.L=R2.CI.L, R2.CI.U=R2.CI.U,
                conf.level=conf.level)
  class(result) <- "mglm.test"
  return(result)
}


