% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sim_lr_prelim.R
\name{sim_lr_prelim}
\alias{sim_lr_prelim}
\title{Simulate Likelihood Ratios from Preliminary Investigation Data}
\usage{
sim_lr_prelim(
  vartype,
  numsims = 1000,
  seed = 123,
  int = 5,
  ErrorRate = 0.05,
  alphaBdate = c(1, 4, 60, 11, 6, 4, 4),
  numReg = 6,
  MP = NULL,
  database = NULL,
  cuts = c(-120, -30, 30, 120, 240, 360)
)
}
\arguments{
\item{vartype}{Character. Type of preliminary investigation variable.
Options: "sex", "region", "age", "height", "birthdate".}

\item{numsims}{Integer. Number of simulations to perform. Default: 1000.}

\item{seed}{Integer. Random seed for reproducibility. Default: 123.}

\item{int}{Numeric. Interval parameter for "age" and "height" variables.
Defines the estimation range (e.g., if MP age is 55 and int is 10,
the range is 45-65). Default: 5.}

\item{ErrorRate}{Numeric (0-1). Error rate for observations. Default: 0.05.}

\item{alphaBdate}{Numeric vector. Alpha parameters for Dirichlet distribution
used in birthdate LR calculations. Usually frequencies of solved cases
in each category. Default: c(1, 4, 60, 11, 6, 4, 4).}

\item{numReg}{Integer. Number of regions in the case (for "region" variable).
Default: 6.}

\item{MP}{Value of the MP's characteristic for closed search. If NULL,
open search is performed. For "sex": "F" or "M"; for "age"/"height":
numeric; for "birthdate": date string; for "region": region ID.
Default: NULL.}

\item{database}{Data frame. Database of POIs for closed search (when MP
is not NULL). Should have columns matching the variable type (e.g.,
"Sex", "Age", "Height", "Region", "DBD"). Can be output from
\code{\link{sim_poi_prelim}}.}

\item{cuts}{Numeric vector. Cutoff values for birthdate categories.
Days difference between declared and actual birth dates.
Default: c(-120, -30, 30, 120, 240, 360).}
}
\value{
A data.frame with two columns:
\itemize{
\item \code{Unrelated}: LR values simulated under H2 (POI is not MP)
\item \code{Related}: LR values simulated under H1 (POI is MP)
}
Each column contains \code{numsims} values.
}
\description{
Simulates likelihood ratio (LR) distributions based on non-genetic
(preliminary investigation) data such as sex, age, region, height, or
birth date. This function generates expected LR distributions under both
hypotheses:
\itemize{
\item H1 (Related): The unidentified person IS the missing person
\item H2 (Unrelated): The unidentified person is NOT the missing person
}
}
\details{
\strong{Open Search (MP = NULL):}
Used when it's unknown whether the MP is in the database. LR is computed
using general population frequencies as the denominator.

\strong{Closed Search (MP specified):}
Used when comparing a specific MP against a database. LR denominator
uses frequencies from the actual database.

\strong{Variable-specific calculations:}
\itemize{
\item \emph{sex}: Binary match/mismatch with error rate
\item \emph{region}: Match against numReg possible regions
\item \emph{age/height}: Match if within +/- int of MP value
\item \emph{birthdate}: Dirichlet-based probability for date discrepancy
}
}
\examples{
# Open search for sex variable
lr_sex <- sim_lr_prelim("sex", numsims = 500, seed = 123)
head(lr_sex)

# Check distribution
summary(log10(lr_sex$Related))
summary(log10(lr_sex$Unrelated))

# Visualize
plot_lr_distribution(lr_sex)

# Closed search with database
db <- sim_poi_prelim(numsims = 100, seed = 456)
lr_sex_closed <- sim_lr_prelim(
  "sex",
  numsims = 500,
  MP = "F",
  database = db
)

# Age variable
lr_age <- sim_lr_prelim("age", numsims = 500, int = 10)
}
\references{
Marsico FL, et al. (2023). "Likelihood ratios for non-genetic evidence
in missing person cases." \emph{Forensic Science International: Genetics},
66, 102891. \doi{10.1016/j.fsigen.2023.102891}
}
\seealso{
\code{\link{sim_lr_genetic}} for genetic LR simulations,
\code{\link{lr_combine}} for combining LRs from different sources,
\code{\link{sim_poi_prelim}} for creating preliminary databases.
}
