% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/densityMclustBounded.R
\name{densityMclustBounded}
\alias{densityMclustBounded}
\alias{print.densityMclustBounded}
\alias{summary.densityMclustBounded}
\alias{print.summary.densityMclustBounded}
\title{Model-based mixture density estimation for bounded data}
\usage{
densityMclustBounded(
  data,
  G = NULL,
  modelNames = NULL,
  criterion = c("BIC", "ICL"),
  lbound = NULL,
  ubound = NULL,
  lambda = c(-3, 3),
  prior = NULL,
  initialization = NULL,
  nstart = 25,
  parallel = FALSE,
  seed = NULL,
  ...
)

\method{summary}{densityMclustBounded}(object, parameters = FALSE, ...)
}
\arguments{
\item{data}{A numeric vector, matrix, or data frame of observations. If a
matrix or data frame, rows correspond to observations and columns correspond
to variables.}

\item{G}{An integer vector specifying the numbers of mixture components. By
default \code{G=1:3}.}

\item{modelNames}{A vector of character strings indicating the Gaussian
mixture models to be fitted on the transformed-data space.  See
\code{\link[mclust:mclustModelNames]{mclust::mclustModelNames()}} for a descripton of available models.}

\item{criterion}{A character string specifying the information criterion for
model selection. Possible values are \code{BIC} (default) or \code{ICL}.}

\item{lbound}{Numeric vector proving lower bounds for variables.}

\item{ubound}{Numeric vector proving upper bounds for variables.}

\item{lambda}{A numeric vector providing the range (min and max) of searched
values for the transformation parameter(s). If a matrix is provided, then
for each variable a row should be provided containing the range of lambda
values for the transformation parameter. If a variable must have a fixed
lambda value, the provided min and max values should be equal. See examples
below.}

\item{prior}{A function specifying a prior for Bayesian regularization of
Gaussian mixtures. See \code{\link[mclust:priorControl]{mclust::priorControl()}} for details.}

\item{initialization}{A list containing one or more of the following
components:
\itemize{
\item \code{noise} A logical or numeric vector indicating an initial guess as to
which observations are noise in the data. If numeric the entries should
correspond to row indexes of the data. If logical an automatic
entropy-based guess of noisy observations is made. When supplied, a noise
term will be added to the model in the estimation.
\item \code{Vinv} When a noise component is included in the model, this is a
numerical optional argument providing the reciprocal of the volume of the
data.  By default, the \code{\link[mclust:hypvol]{mclust::hypvol()}} is used on the transformed data
from a preliminary model.
}}

\item{nstart}{An integer value specifying the number of replications of
k-means clustering to be used for initializing the EM algorithm. See
\code{\link[=kmeans]{kmeans()}}.}

\item{parallel}{An optional argument which allows to specify if the search
over all possible models should be run sequentially (default) or in
parallel.

For a single machine with multiple cores, possible values are:
\itemize{
\item a logical value specifying if parallel computing should be used (\code{TRUE})
or not (\code{FALSE}, default) for evaluating the fitness function;
\item a numerical value which gives the number of cores to employ. By default,
this is obtained from the function \code{\link[parallel:detectCores]{parallel::detectCores()}};
\item a character string specifying the type of parallelisation to use. This
depends on system OS: on Windows OS only \code{"snow"} type functionality is
available, while on Unix/Linux/Mac OSX both \code{"snow"} and \code{"multicore"}
(default) functionalities are available.
}

In all the cases described above, at the end of the search the cluster is
automatically stopped by shutting down the workers.

If a cluster of multiple machines is available, evaluation of the fitness
function can be executed in parallel using all, or a subset of, the cores
available to the machines belonging to the cluster. However, this option
requires more work from the user, who needs to set up and register a
parallel back end.  In this case the cluster must be explicitely stopped
with \code{\link[parallel:makeCluster]{parallel::stopCluster()}}.}

\item{seed}{An integer value containing the random number generator state.
This argument can be used to replicate the result of k-means initialisation
strategy. Note that if parallel computing is required, the \pkg{doRNG}
package must be installed.}

\item{\dots}{Further arguments passed to or from other methods.}

\item{object}{An object of class \code{'densityMclustBounded'}.}

\item{parameters}{A logical, if \code{TRUE} the estimated parameters of mixture
components are printed.}
}
\value{
Returns an object of class \code{'densityMclustBounded'}.
}
\description{
Density estimation for bounded data via transformation-based approach for
Gaussian mixtures.
}
\details{
For more details see
\code{vignette("mclustAddons")}
}
\examples{
\donttest{
# univariate case with lower bound
x <- rchisq(200, 3)
xgrid <- seq(-2, max(x), length=1000)
f <- dchisq(xgrid, 3)  # true density
dens <- densityMclustBounded(x, lbound = 0)
summary(dens)
summary(dens, parameters = TRUE)
plot(dens, what = "BIC")
plot(dens, what = "density")
lines(xgrid, f, lty = 2)
plot(dens, what = "density", data = x, breaks = 15)

# univariate case with lower & upper bounds
x <- rbeta(200, 5, 1.5)
xgrid <- seq(-0.1, 1.1, length=1000)
f <- dbeta(xgrid, 5, 1.5)  # true density
dens <- densityMclustBounded(x, lbound = 0, ubound = 1)
summary(dens)
plot(dens, what = "BIC")
plot(dens, what = "density")
plot(dens, what = "density", data = x, breaks = 9)

# bivariate case with lower bounds
x1 <- rchisq(200, 3)
x2 <- 0.5*x1 + sqrt(1-0.5^2)*rchisq(200, 5)
x <- cbind(x1, x2)
plot(x)
dens <- densityMclustBounded(x, lbound = c(0,0))
summary(dens, parameters = TRUE)
plot(dens, what = "BIC")
plot(dens, what = "density")
plot(dens, what = "density", type = "hdr")
plot(dens, what = "density", type = "persp")
# specify different ranges for the lambda values of each variable
dens1 <- densityMclustBounded(x, lbound = c(0,0), 
                              lambda = matrix(c(-2,2,0,1), 2, 2, byrow=TRUE))
# set lambda = 0 fixed for the second variable
dens2 <- densityMclustBounded(x, lbound = c(0,0), 
                              lambda = matrix(c(0,1,0,0), 2, 2, byrow=TRUE))

dens[c("lambdaRange", "lambda", "loglik", "df")]
dens1[c("lambdaRange", "lambda", "loglik", "df")]
dens2[c("lambdaRange", "lambda", "loglik", "df")]
}

}
\references{
Scrucca L. (2019) A transformation-based approach to Gaussian
mixture density estimation for bounded data. \emph{Biometrical Journal},
61:4, 873–888. \doi{doi:10.1002/bimj.201800174}
}
\seealso{
\code{\link[=predict.densityMclustBounded]{predict.densityMclustBounded()}}, \code{\link[=plot.densityMclustBounded]{plot.densityMclustBounded()}}.
}
\author{
Luca Scrucca
}
