% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/filter.R
\name{filter_column}
\alias{filter_column}
\title{Subset columns using annotation values}
\usage{
filter_column(.ms, ..., .preserve = FALSE)
}
\arguments{
\item{.ms}{\code{matrixset} object to subset based on the filtering
conditions}

\item{...}{Condition, or expression, that returns a logical value,
used to determine if columns are kept or discarded. The
expression may refer to column annotations - columns of
the \code{column_info} component of \code{.ms} More than one
condition can be supplied and if multiple
expressions are included, they are combined with the \code{&}
operator. Only columns for which all conditions evaluate
to TRUE are kept.}

\item{.preserve}{\code{logical}, relevant only if \code{.ms} is column grouped. When
\code{.preserve} is \code{FALSE} (the default), the column grouping
is updated based on the new \code{matrixset} resulting from
the filtering. Otherwise, the column grouping is kept as
is.}
}
\value{
A \code{matrixset}, with possibly a subset of the columns of the original object.
Groups will be updated if \code{.preserve} is \code{TRUE}.
}
\description{
The \code{\link[=filter_column]{filter_column()}} function subsets the columns of all matrices of a
\code{matrixset}, retaining all columns that satisfy given condition(s). The
function \code{filter_column} works like \code{dplyr}'s \code{\link[dplyr:filter]{dplyr::filter()}}.
}
\details{
The conditions are given as expressions in \code{...}, which are applied to
columns of the annotation data frame (\code{column_info}) to determine which
columns should be retained.

It can be applied to both grouped and ungrouped \code{matrixset} (see
\code{\link[=column_group_by]{column_group_by()}}), and section \sQuote{Grouped matrixsets}.
}
\section{Grouped matrixsets}{

Row grouping (\code{\link[=row_group_by]{row_group_by()}}) has no impact on column filtering.

The impact of column grouping (\code{\link[=column_group_by]{column_group_by()}}) on column filtering
depends on the conditions. Often, column grouping will not have any impact,
but as soon as an aggregating, lagging or ranking function is involved, then
the results will differ.

For instance, the two following are not equivalent (except by pure
coincidence).

\code{student_results \%>\% filter_column(school_average > mean(school_average))}

And it's grouped equivalent:
\code{student_results \%>\% column_group_by(program) \%>\% filter_column(school_average > mean(school_average))}

In the ungrouped version, the mean of \code{school_average} is taken globally
and \code{filter_column} keeps columns with \code{school_average} greater than this
global average. In the grouped version, the average is calculated within each
\code{class} and the kept columns are the ones with \code{school_average} greater
than the within-class average.
}

\examples{
# Filtering using one condition
filter_column(student_results, program == "Applied Science")

# Filetring using multiple conditions. These are equivalent
filter_column(student_results, program == "Applied Science" & school_average > 0.8)
filter_column(student_results, program == "Applied Science", school_average > 0.8)

# The potential difference between grouped and non-grouped.
filter_column(student_results, school_average > mean(school_average))
sr <- column_group_by(student_results, program)
filter_column(sr, school_average > mean(school_average))

}
