#' Generic plot function for bcbnp object.
#'
#' This function is built with `ggplot2`. Parameters such as `binwidth`, `adjust`, `density.color`, `density.lwd`, and `density.lty` correspond directly to arguments in the `ggplot2` geoms.
#'
#' @param x The object generated by the bcbnp function (the BC-BNP model output).
#' @param plot.type A character string specifying the type of plot:
#' - "predictive": Plots the posterior predictive distribution of a new study.
#' - "mu_posterior": Plots the posterior of the overall mean (mu.theta).
#' - "tau_posterior": Plots the posterior of the between-study standard deviation (tau.theta).
#' @param n.samples The number of posterior samples to plot. If NULL, all samples are used.
#' @param x.lim Numeric vector of length 2 specifying the x-axis limits.
#' @param y.lim Numeric vector of length 2 specifying the y-axis limits.
#' @param x.lab Text with the label of the x-axis.
#' @param y.lab Text with the label of the y-axis.
#' @param title.plot Text for setting a title in the plot.
#' @param show.histogram A logical value indicating whether to show the histogram.
#' @param show.rug A logical value indicating whether to show the rug plot. Defaults to TRUE for predictive and mu plots, FALSE for tau.
#' @param binwidth A numeric value specifying the bin width for the histogram.
#' @param adjust A numeric value for the density bandwidth adjustment.
#' @param density.color A character value for the density curve color.
#' @param hist.color A character value for the histogram border color.
#' @param hist.fill A character value for the histogram fill color.
#' @param rug.color A character value for the rug plot color.
#' @param rug.alpha A numeric value between 0 and 1 for the rug plot transparency.
#' @param density.alpha A numeric value for the density curve transparency.
#' @param density.lwd A numeric value for the density curve line width.
#' @param density.lty A numeric value for the density curve line type.
#'
#' @param ... \dots
#'
#' @import ggplot2
#'
#' @export
plot.bcbnp = function(x,
                      plot.type = "predictive",
                      n.samples = NULL,
                      x.lim = NULL,
                      y.lim = NULL,
                      x.lab = NULL,
                      y.lab = "Posterior Predictive Distribution",
                      title.plot = NULL,
                      show.histogram = TRUE,
                      show.rug = NULL,
                      binwidth = 0.08,
                      adjust = 1,
                      density.color = "blue",
                      hist.color = "black",
                      hist.fill = "grey95",
                      rug.color = "green",
                      rug.alpha = 1,
                      density.alpha = 0.2,
                      density.lwd = 1,
                      density.lty = 1,
                      ...) {

  # Set plot-specific defaults if not provided
  if (is.null(show.rug)) {
    show.rug = ifelse(plot.type == "tau_posterior", FALSE, TRUE)
  }

  # Use all samples unless a smaller number is specified
  total_sims = dim(x$BUGSoutput$sims.list$theta)[1]
  if (is.null(n.samples)) {
    n.samples = total_sims
  }

  # Set default titles based on plot.type if title.plot is not provided
  if (is.null(title.plot)) {
    if (plot.type == "predictive") {
      title.plot = "BC-BNP"
    } else if (plot.type == "mu_posterior") {
      title.plot = "Posterior for mu "
    } else if (plot.type == "tau_posterior") {
      title.plot = "Posterior for tau "
    }
  }

  # Set default x-axis labels if not provided
  if (is.null(x.lab)) {
    if (plot.type == "mu_posterior") {
      x.lab = expression(mu[theta])
    } else if (plot.type == "tau_posterior") {
      x.lab = expression(tau[theta])
    } else {
      x.lab = expression(theta^{new})
    }
  }


  # --- Plot Type: Predictive Distribution ---
  if (plot.type == "predictive") {
    N = length(x$data$TE) + 1

    density_data_samples = x$BUGSoutput$sims.list$theta[1:n.samples, N]
    hist_data_samples = x$BUGSoutput$sims.list$theta.bias[1:n.samples, N]

    plot_data_density = data.frame(effect = as.vector(density_data_samples))
    plot_data_hist = data.frame(effect = as.vector(hist_data_samples))

    # Calculate default x.lim if not specified
    if (is.null(x.lim)) {
      q1 = quantile(plot_data_density$effect, 0.25, na.rm = TRUE)
      q3 = quantile(plot_data_density$effect, 0.75, na.rm = TRUE)
      iqr = q3 - q1
      lower_lim = q1 - 4 * iqr
      upper_lim = q3 + 4 * iqr
      x.lim = c(lower_lim, upper_lim)
    }

    # Create the plot
    plot1 = ggplot(data = plot_data_hist, aes(x = effect))

    if (show.histogram) {
      plot1 = plot1 + geom_histogram(aes(y = after_stat(density)), binwidth = binwidth, colour = hist.color, fill = hist.fill)
    }

    plot1 = plot1 + geom_density(data = plot_data_density, color = density.color, lwd = density.lwd, lty = density.lty, alpha = density.alpha, adjust = adjust)

    if (show.rug) {
      rug_data = data.frame(TE = x$data$TE)
      plot1 = plot1 + geom_rug(data = rug_data, aes(x = TE), sides = "b", lwd = 1, col = rug.color, alpha = rug.alpha, show.legend = FALSE)
    }

    plot1 = plot1 +
      coord_cartesian(xlim = x.lim, ylim = y.lim) +  # Replaced scale_x_continuous
      xlab(x.lab) +
      ylab(y.lab) +
      theme_bw() +
      ggtitle(title.plot) +
      theme(legend.position = "none")

    return(suppressWarnings(plot1))

    # --- Plot Type: Mu Posterior ---
  } else if (plot.type == "mu_posterior") {
    mu_samples = x$BUGSoutput$sims.list$mu.theta[1:n.samples]
    plot_data_mu = data.frame(effect = as.vector(mu_samples))

    # Calculate default x.lim if not specified
    if (is.null(x.lim)) {
      q1 = quantile(plot_data_mu$effect, 0.25, na.rm = TRUE)
      q3 = quantile(plot_data_mu$effect, 0.75, na.rm = TRUE)
      iqr = q3 - q1
      lower_lim = q1 - 4 * iqr
      upper_lim = q3 + 4 * iqr
      x.lim = c(lower_lim, upper_lim)
    }

    # Create the plot
    plot_mu = ggplot(data = plot_data_mu, aes(x = effect))

    if (show.histogram) {
      plot_mu = plot_mu + geom_histogram(aes(y = after_stat(density)), binwidth = binwidth, colour = hist.color, fill = hist.fill)
    }

    plot_mu = plot_mu + geom_density(color = density.color, lwd = density.lwd, lty = density.lty, alpha = density.alpha, adjust = adjust)

    if (show.rug) {
      rug_data = data.frame(TE = x$data$TE)
      plot_mu = plot_mu + geom_rug(data = rug_data, aes(x = TE), sides = "b", lwd = 1, col = rug.color, alpha = rug.alpha, show.legend = FALSE)
    }

    plot_mu = plot_mu + geom_vline(aes(xintercept = 0), lty = 2, color = "black")

    plot_mu = plot_mu +
      coord_cartesian(xlim = x.lim, ylim = y.lim) + # Replaced scale_x_continuous
      xlab(x.lab) +
      ylab(y.lab) +
      theme_bw() +
      ggtitle(title.plot) +
      theme(legend.position = "none")

    return(suppressWarnings(plot_mu))

    # --- Plot Type: Tau Posterior ---
  } else if (plot.type == "tau_posterior") {
    tau_samples = x$BUGSoutput$sims.list$tau.theta[1:n.samples]
    plot_data_tau = data.frame(effect = as.vector(tau_samples))

    # Calculate default x.lim if not specified
    if (is.null(x.lim)) {
      q1 = quantile(plot_data_tau$effect, 0.25, na.rm = TRUE)
      q3 = quantile(plot_data_tau$effect, 0.75, na.rm = TRUE)
      iqr = q3 - q1
      # Note: Tau should not be negative, so the lower bound should be max(0, q1 - 4 * iqr)
      lower_lim = max(0, q1 - 4 * iqr)
      upper_lim = q3 + 4 * iqr
      x.lim = c(lower_lim, upper_lim)
    }

    # Create the plot
    plot_tau = ggplot(data = plot_data_tau, aes(x = effect))

    if (show.histogram) {
      plot_tau = plot_tau + geom_histogram(aes(y = after_stat(density)), binwidth = binwidth, colour = hist.color, fill = hist.fill)
    }

    plot_tau = plot_tau + geom_density(color = density.color, lwd = density.lwd, lty = density.lty, alpha = density.alpha, adjust = adjust)

    if (show.rug) {
      rug_data = data.frame(TE = x$data$TE)
      plot_tau = plot_tau + geom_rug(data = rug_data, aes(x = TE), sides = "b", lwd = 1, col = rug.color, alpha = rug.alpha, show.legend = FALSE)
    }

    plot_tau = plot_tau + geom_vline(aes(xintercept = 0), lty = 2, color = "black")

    plot_tau = plot_tau +
      coord_cartesian(xlim = x.lim, ylim = y.lim) +
      xlab(x.lab) +
      ylab(y.lab) +
      theme_bw() +
      ggtitle(title.plot) +
      theme(legend.position = "none")

    return(suppressWarnings(plot_tau))

  } else {
    return(invisible(NULL))
  }
}
