#include <Rcpp.h>
using namespace Rcpp;


// Function that accepts parameter values for 3 main effects and two interaction
// and performs the sobel test. Main effects and covariance terms required.

//' Sobel test for the indirect effect - Two ordered mediator path
//'
//' This function takes the parameter estimates and covariances
//' and performs the Sobel test for two ordered mediators.
//'
//' @param mu1 Value of the estimate of the independent variable on the first mediator.
//' @param sig1 Value of the variance of the estimate of the independent variable on the first mediator.
//' @param mu2 Value of the estimate of the first mediator on the response.  
//' @param sig2 Value of the variance of the estimate of the first mediator on the response.  
//' @param mu3 Value of the estimate of the independent variable on the second mediator.
//' @param sig3 Value of the variance of the estimate of the independent variable on the second mediator.
//' @param mu4 Value of the estimate of the second mediator on the response.  
//' @param sig4 Value of the variance of the estimate of the second mediator on the response.  
//' @param mu2b Value of the estimate of the effect of the interaction of the independent and first mediator variable on the response.
//' @param sig2b Value of the variance of the estimate of the effect of the interaction of the independent and first mediator variable on the response.
//' @param mu4b Value of the estimate of the effect of the interaction of the independent and second mediator variable on the response. 
//' @param sig4b Value of the variance of the estimate of the effect of the interaction of the independent and second mediator variable on the response.
//' @param sig12 Value of the covariance between mu1 and mu2.
//' @param sig12b Value of the covariance between mu1 and mu2b. 
//' @param sig13 Value of the covariance between mu1 and mu3. 
//' @param sig14 Value of the covariance between mu1 and mu4. 
//' @param sig14b Value of the covariance between mu1 and mu4b.
//' @param sig22b Value of the covariance between mu1 and mu2b. 
//' @param sig23 Value of the covariance between mu2 and mu3. 
//' @param sig24 Value of the covariance between mu2 and mu4. 
//' @param sig24b Value of the covariance between mu2 and mu4b.
//' @param sig2b3 Value of the covariance between mu2b and mu3. 
//' @param sig2b4 Value of the covariance between mu2b and mu4. 
//' @param sig2b4b Value of the covariance between mu2b and mu4b.
//' @param sig34 Value of the covariance between mu3 and mu4. 
//' @param sig34b Value of the covariance between mu3 and mu4b.
//' @param sig44b Value of the covariance between mu4 and mu4b.
//' @param indL Value indicating the value of the independent variable used for the interaction. Typically 1.
//' @param mu1_0 Null value for mu1.
//' @param mu2_0 Null value for mu2.
//' @param mu3_0 Null value for mu3.
//' @param mu4_0 Null value for mu4.
//' @param mu2b_0 Null value for mu2b.
//' @param mu4b_0 Null value for mu4b.
//' @returns A p-value for the test for the indirect effect.
//' @export
//' @examples
//' sobelTest_unord(1, .1, .25, .01, 0, 0, .15, .01, 0, 0, 0, 0)
// [[Rcpp::export]]
double sobelTest_unord(const long double & mu1, const long double & sig1, // alpha1
                         const long double & mu2, const long double & sig2, // gamma1
                         const long double & mu3, const long double & sig3, // alpha2
                         const long double & mu4, const long double & sig4, // gamma2
                         
                         const long double & mu2b, const long double & sig2b, // psi1
                         const long double & mu4b, const long double & sig4b, //psi2

                         const long double & sig12 = 0, const long double & sig12b = 0,
                         const long double & sig13 = 0,
                         const long double & sig14 = 0, const long double & sig14b = 0,

                         const long double & sig22b = 0, const long double & sig23 = 0,
                         const long double & sig24 = 0, const long double & sig24b = 0,

                         const long double & sig2b3 = 0,
                         const long double & sig2b4 = 0, const long double & sig2b4b = 0,

                         const long double & sig34 = 0, const long double & sig34b = 0,

                         const long double & sig44b = 0,

                         const int & indL = 1,
                         double mu1_0 = 0, double mu2_0 = 0, 
                         double mu3_0 = 0, double mu4_0 = 0, 
                         double mu2b_0 = 0, double mu4b_0 = 0){

  // Delta = alpha1(gamma1 + psi1*G) + alpha2(gamma2 + psi2*G);
  // alpha1 = mu1, gamma1 = mu2, psi1 = mu2b
  // alpha2 = mu3, gamma2 = mu4, psi2 = mu4b

  // g1 = derivative of delta wrt alpha1, g2 for gamma1, g3 for psi1,
  // g4 = derivative of delta wrt alpha2, g5 for gamma2, g6 for psi2,
  double g1 = mu2+ mu2b*indL;
  double g2 = mu1;
  double g3 = mu1*indL;
  double g4 = mu4+ mu4b*indL;
  double g5 = mu3;
  double g6 = mu3*indL;

  double p1 = g1*sig1   + g2*sig12  + g3*sig12b  + g4*sig13  + g5*sig14  + g6*sig14b;
  double p2 = g1*sig12  + g2*sig2   + g3*sig22b  + g4*sig23  + g5*sig24  + g6*sig24b;
  double p3 = g1*sig12b + g2*sig22b + g3*sig2b   + g4*sig2b3 + g5*sig2b4 + g6*sig2b4b;
  double p4 = g1*sig13  + g2*sig23  + g3*sig2b3  + g4*sig3   + g5*sig34  + g6*sig34b;
  double p5 = g1*sig14  + g2*sig24  + g3*sig2b4  + g4*sig34  + g5*sig4   + g6*sig44b;
  double p6 = g1*sig14b + g2*sig24b + g3*sig2b4b + g4*sig34b + g5*sig44b + g6*sig4b;

  double deltaSE = std::sqrt( g1*p1 + g2*p2 + g3*p3 + g4*p4 + g5*p5 + g6*p6 );

  double deltaEst = mu1*(mu2+indL*mu2b) + mu3*(mu4+indL*mu4b);
  double deltaNull = mu1_0*(mu2_0+indL*mu2b_0) + mu3_0*(mu4_0+indL*mu4b_0);
  return R::pnorm(
    -fabs(
        (deltaEst - deltaNull) / deltaSE) , 0, 1, 1, 0)*2.0;
}
