% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/format_data.R
\name{fmt_spelled_num}
\alias{fmt_spelled_num}
\title{Format values to spelled-out numbers}
\usage{
fmt_spelled_num(
  data,
  columns = everything(),
  rows = everything(),
  pattern = "{x}",
  locale = NULL
)
}
\arguments{
\item{data}{\emph{The gt table data object}

\verb{obj:<gt_tbl>} // \strong{required}

This is the \strong{gt} table object that is commonly created through use of the
\code{\link[=gt]{gt()}} function.}

\item{columns}{\emph{Columns to target}

\code{\link[=rows-columns]{<column-targeting expression>}} // \emph{default:} \code{everything()}

Can either be a series of column names provided in \code{c()}, a vector of
column indices, or a select helper function (e.g. \code{\link[=starts_with]{starts_with()}},
\code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}}, \code{\link[=num_range]{num_range()}} and \code{\link[=everything]{everything()}}).}

\item{rows}{\emph{Rows to target}

\code{\link[=rows-columns]{<row-targeting expression>}} // \emph{default:} \code{everything()}

In conjunction with \code{columns}, we can specify which of their rows should
undergo formatting. The default \code{\link[=everything]{everything()}} results in all rows in
\code{columns} being formatted. Alternatively, we can supply a vector of row
captions within \code{c()}, a vector of row indices, or a select helper
function (e.g. \code{\link[=starts_with]{starts_with()}}, \code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}},
\code{\link[=num_range]{num_range()}}, and \code{\link[=everything]{everything()}}). We can also use expressions to filter
down to the rows we need (e.g., \verb{[colname_1] > 100 & [colname_2] < 50}).}

\item{pattern}{\emph{Specification of the formatting pattern}

\verb{scalar<character>} // \emph{default:} \code{"{x}"}

A formatting pattern that allows for decoration of the formatted value. The
formatted value is represented by the \code{{x}} (which can be used multiple
times, if needed) and all other characters will be interpreted as string
literals.}

\item{locale}{\emph{Locale identifier}

\verb{scalar<character>} // \emph{default:} \code{NULL} (\code{optional})

An optional locale identifier that can be used for formatting values
according to the locale's rules. Examples include \code{"en"} for English (United
States) and \code{"fr"} for French (France). We can call \code{\link[=info_locales]{info_locales()}} for a
useful reference for all of the locales that are supported. A locale ID can
be also set in the initial \code{\link[=gt]{gt()}} function call (where it would be used
automatically by any function with a \code{locale} argument) but a \code{locale}
value provided here will override that global locale.}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
With numeric values in a \strong{gt} table we can transform those to numbers that
are spelled out with \code{fmt_spelled_num()}. Any values from \code{0} to \code{100} can be
spelled out so, for example, the value \code{23} will be formatted as
\code{"twenty-three"}. Providing a locale ID will result in the number spelled out
in the locale's language rules. For example, should a Swedish locale (\code{"sv"})
be provided, the value \code{23} will yield \code{"tjugotre"}. In addition to this, we
can optionally use the \code{pattern} argument for decoration of the formatted
values.
}
\section{Compatibility of formatting function with data values}{


\code{fmt_spelled_num()} is compatible with body cells that are of the \code{"numeric"}
or \code{"integer"} types. Any other types of body cells are ignored during
formatting. This is to say that cells of incompatible data types may be
targeted, but there will be no attempt to format them.
}

\section{Compatibility of arguments with the \code{from_column()} helper function}{


\code{\link[=from_column]{from_column()}} can be used with certain arguments of \code{fmt_spelled_num()} to
obtain varying parameter values from a specified column within the table.
This means that each row could be formatted a little bit differently. These
arguments provide support for \code{\link[=from_column]{from_column()}}:
\itemize{
\item \code{pattern}
\item \code{locale}
}

Please note that for both of the aforementioned arguments, a \code{\link[=from_column]{from_column()}}
call needs to reference a column that has data of the correct type (this is
different for each argument). Additional columns for parameter values can be
generated with \code{\link[=cols_add]{cols_add()}} (if not already present). Columns that contain
parameter data can also be hidden from final display with \code{\link[=cols_hide]{cols_hide()}}.
Finally, there is no limitation to how many arguments the \code{\link[=from_column]{from_column()}}
helper is applied so long as the arguments belong to this closed set.
}

\section{Supported locales}{


The following 80 locales are supported in the \code{locale} argument of
\code{fmt_spelled_num()}: \code{"af"} (Afrikaans), \code{"ak"} (Akan), \code{"am"} (Amharic),
\code{"ar"} (Arabic), \code{"az"} (Azerbaijani), \code{"be"} (Belarusian), \code{"bg"}
(Bulgarian), \code{"bs"} (Bosnian), \code{"ca"} (Catalan), \code{"ccp"} (Chakma), \code{"chr"}
(Cherokee), \code{"cs"} (Czech), \code{"cy"} (Welsh), \code{"da"} (Danish), \code{"de"} (German),
\code{"de-CH"} (German (Switzerland)), \code{"ee"} (Ewe), \code{"el"} (Greek), \code{"en"}
(English), \code{"eo"} (Esperanto), \code{"es"} (Spanish), \code{"et"} (Estonian), \code{"fa"}
(Persian), \code{"ff"} (Fulah), \code{"fi"} (Finnish), \code{"fil"} (Filipino), \code{"fo"}
(Faroese), \code{"fr"} (French), \code{"fr-BE"} (French (Belgium)), \code{"fr-CH"} (French
(Switzerland)), \code{"ga"} (Irish), \code{"he"} (Hebrew), \code{"hi"} (Hindi), \code{"hr"}
(Croatian), \code{"hu"} (Hungarian), \code{"hy"} (Armenian), \code{"id"} (Indonesian),
\code{"is"} (Icelandic), \code{"it"} (Italian), \code{"ja"} (Japanese), \code{"ka"} (Georgian),
\code{"kk"} (Kazakh), \code{"kl"} (Kalaallisut), \code{"km"} (Khmer), \code{"ko"} (Korean),
\code{"ky"} (Kyrgyz), \code{"lb"} (Luxembourgish), \code{"lo"} (Lao), \code{"lrc"} (Northern
Luri), \code{"lt"} (Lithuanian), \code{"lv"} (Latvian), \code{"mk"} (Macedonian), \code{"ms"}
(Malay), \code{"mt"} (Maltese), \code{"my"} (Burmese), \code{"ne"} (Nepali), \code{"nl"} (Dutch),
\code{"nn"} (Norwegian Nynorsk), \code{"no"} (Norwegian), \code{"pl"} (Polish), \code{"pt"}
(Portuguese), \code{"qu"} (Quechua), \code{"ro"} (Romanian), \code{"ru"} (Russian), \code{"se"}
(Northern Sami), \code{"sk"} (Slovak), \code{"sl"} (Slovenian), \code{"sq"} (Albanian),
\code{"sr"} (Serbian), \code{"sr-Latn"} (Serbian (Latin)), \code{"su"} (Sundanese), \code{"sv"}
(Swedish), \code{"sw"} (Swahili), \code{"ta"} (Tamil), \code{"th"} (Thai), \code{"tr"} (Turkish),
\code{"uk"} (Ukrainian), \code{"vi"} (Vietnamese), \code{"yue"} (Cantonese), and \code{"zh"}
(Chinese).
}

\section{Examples}{


Let's use a summarized version of the \code{\link{gtcars}} dataset to create a
\strong{gt} table. \code{fmt_spelled_num()} is used to transform
integer values into spelled-out numbering (in the \code{n} column). That formatted
column of numbers-as-words is given cell background colors via \code{\link[=data_color]{data_color()}}
(the underlying numerical values are always available).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{gtcars |>
  dplyr::count(mfr, ctry_origin) |>
  dplyr::arrange(ctry_origin) |>
  gt(rowname_col = "mfr", groupname_col = "ctry_origin") |>
  cols_label(n = "No. of Entries") |>
  fmt_spelled_num() |>
  tab_stub_indent(rows = everything(), indent = 2) |>
  data_color(
    columns = n,
    method = "numeric",
    palette = "viridis",
    alpha = 0.8
  ) |>
  opt_all_caps() |>
  opt_vertical_padding(scale = 0.5) |>
  cols_align(align = "center", columns = n)
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_spelled_num_1.png" alt="This image of a table was generated from the first code example in the `fmt_spelled_num()` help file." style="width:100\%;">
}}

With a considerable amount of \strong{dplyr} and \strong{tidyr} work done to the
\code{\link{pizzaplace}} dataset, we can create a new \strong{gt} table. \code{fmt_spelled_num()}
will be used here to transform the integer values in the \code{rank} column.
We'll do so with a special \code{pattern} that puts the word 'Number' in front of
every spelled-out number.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{pizzaplace |>
  dplyr::mutate(month = lubridate::month(date, label = TRUE)) |>
  dplyr::filter(month \%in\% month.abb[1:6]) |>
  dplyr::group_by(name, month) |>
  dplyr::summarize(sum = sum(price), .groups = "drop") |>
  dplyr::arrange(month, desc(sum)) |>
  dplyr::group_by(month) |>
  dplyr::slice_head(n = 5) |>
  dplyr::mutate(rank = dplyr::row_number()) |>
  dplyr::ungroup() |>
  dplyr::select(-sum) |>
  tidyr::pivot_wider(names_from = month, values_from = c(name)) |>
  gt() |>
  fmt_spelled_num(columns = rank, pattern = "Number \{x\}") |>
  opt_all_caps() |>
  cols_align(columns = -rank, align = "center") |>
  cols_width(
    rank ~ px(120),
    everything() ~ px(100)
  ) |>
  opt_table_font(stack = "rounded-sans") |>
  tab_options(table.font.size = px(14))
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_spelled_num_2.png" alt="This image of a table was generated from the second code example in the `fmt_spelled_num()` help file." style="width:100\%;">
}}

Let's make a table that compares how the numbers from \code{1} to \code{10} are spelled
across a small selection of languages. Here we use \code{fmt_spelled_num()} with
each column, ensuring that the \code{locale} value matches that of the column
name.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{dplyr::tibble(
  num = 1:10,
  en = num,
  fr = num,
  de = num,
  es = num,
  pl = num,
  bg = num,
  ko = num,
  zh = num
) |>
  gt(rowname_col = "num") |>
  fmt_spelled_num(columns = en, locale = "en") |>
  fmt_spelled_num(columns = fr, locale = "fr") |>
  fmt_spelled_num(columns = de, locale = "de") |>
  fmt_spelled_num(columns = es, locale = "es") |>
  fmt_spelled_num(columns = pl, locale = "pl") |>
  fmt_spelled_num(columns = bg, locale = "bg") |>
  fmt_spelled_num(columns = ko, locale = "ko") |>
  fmt_spelled_num(columns = zh, locale = "zh") |>
  cols_label_with(fn = function(x) md(paste0("`", x, "`"))) |>
  tab_spanner(
    label = "Numbers in the specified locale",
    columns = everything()
  ) |>
  cols_align(align = "left", columns = everything()) |>
  cols_width(
    c(en, fr, de, es, pl, bg) ~ px(100),
    c(ko, zh) ~ px(50)
  ) |>
  opt_horizontal_padding(scale = 2) |>
  opt_vertical_padding(scale = 0.5)
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_spelled_num_3.png" alt="This image of a table was generated from the third code example in the `fmt_spelled_num()` help file." style="width:100\%;">
}}
}

\section{Function ID}{

3-12
}

\section{Function Introduced}{

\code{v0.9.0} (Mar 31, 2023)
}

\seealso{
The vector-formatting version of this function:
\code{\link[=vec_fmt_spelled_num]{vec_fmt_spelled_num()}}.

Other data formatting functions: 
\code{\link{data_color}()},
\code{\link{fmt}()},
\code{\link{fmt_auto}()},
\code{\link{fmt_bins}()},
\code{\link{fmt_bytes}()},
\code{\link{fmt_chem}()},
\code{\link{fmt_country}()},
\code{\link{fmt_currency}()},
\code{\link{fmt_date}()},
\code{\link{fmt_datetime}()},
\code{\link{fmt_duration}()},
\code{\link{fmt_email}()},
\code{\link{fmt_engineering}()},
\code{\link{fmt_flag}()},
\code{\link{fmt_fraction}()},
\code{\link{fmt_icon}()},
\code{\link{fmt_image}()},
\code{\link{fmt_index}()},
\code{\link{fmt_integer}()},
\code{\link{fmt_markdown}()},
\code{\link{fmt_number}()},
\code{\link{fmt_number_si}()},
\code{\link{fmt_partsper}()},
\code{\link{fmt_passthrough}()},
\code{\link{fmt_percent}()},
\code{\link{fmt_roman}()},
\code{\link{fmt_scientific}()},
\code{\link{fmt_tf}()},
\code{\link{fmt_time}()},
\code{\link{fmt_units}()},
\code{\link{fmt_url}()},
\code{\link{sub_large_vals}()},
\code{\link{sub_missing}()},
\code{\link{sub_small_vals}()},
\code{\link{sub_values}()},
\code{\link{sub_zero}()}
}
\concept{data formatting functions}
