% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulation.R
\name{glmstarma.sim}
\alias{glmstarma.sim}
\title{Simulate spatial time-series based on generalized linear models}
\usage{
glmstarma.sim(
  ntime,
  parameters,
  model,
  family = NULL,
  wlist,
  covariates = list(),
  wlist_past_mean = NULL,
  wlist_covariates = NULL,
  n_start = 100L,
  control = list()
)
}
\arguments{
\item{ntime}{Number of observation times to be simulated}

\item{parameters}{a named list specifying the parameters of the model to be simulated, which has the following elements:
\itemize{
\item \code{intercept} (numeric): Intercept parameter. If an inhomogeneous model is simulated, a value must be specified for each component of the time series.
\item \code{past_obs} (numeric matrix): Parameter values for the past observations.
\item \code{past_mean} (numeric matrix): Parameter values for the past means.
\item \code{covariates} (numeric matrix): Parameter values for the covariates.
}}

\item{model}{a named list specifying the model for the linear predictor, which can be of the following elements:
\itemize{
\item \code{intercept} (character): 'homogenous' (default) for a homogenous model, i.e. the same intercept for all components, 'inhomogenous' for inhomogenous models, i.e. an individual intercept for each component.
\item \code{past_obs} (integer vector/binary matrix): Maximal spatial orders for the time lags in \code{past_obs_time_lags}. A binary matrix can be passed as an alternative, with the entry in row \eqn{i} and column \eqn{j} indicating whether the \eqn{(i - 1)}-spatial lag for the \eqn{j}-th time lag is included in the model. If not specified, no regression on past observations is performed.
\item \code{past_obs_time_lags} (optional integer vector) indicates the time lags for regression on past observations. Defaults are \code{seq(length(past_obs))} (for vectors) and \code{seq(ncol(past_obs))} (for a matrix)
\item \code{past_mean} (integer vector/binary matrix): Spatial orders for the regression on the (latent) feedback process. Values can be entered in the same format as in \code{past_obs}. If not specified, no regression to the feedback process is performed.
\item \code{past_mean_time_lags} (optional integer vector) indicates the time lags for regression on past values of the feedback process. Defaults are \code{seq(length(past_mean))} (for vectors) and \code{seq(ncol(past_mean))} (for a matrix)
\item \code{covariates} (integer vector/binary matrix) spatial orders for the covariate processes passed in the argument \code{covariates}. The values can be passed as in \code{past_obs} and \code{past_means}, where the jth entry or column represents the jth covariable. If no values are specified but covariates are included, the spatial order 0 is used by default, which corresponds to the first matrix in argument \code{wlist_covariates}.
}}

\item{family}{An object of class \code{stfamily} that specifies the marginal distributions and the type of model fitted.}

\item{wlist}{A list of quadratic matrices, with the same dimension as the time series, which describe the spatial dependencies. Row-normalized matrices are recommended.}

\item{covariates}{List of covariates, containing matrices or returns of the covariate functions of this package (see also \code{\link{TimeConstant}}, \code{\link{SpatialConstant}}). The matrices must have the same dimension as \code{ts}}

\item{wlist_past_mean}{(Optional) List of matrices, which describes spatial dependencies for the past mean. If this is \code{NULL}, the matrices from \code{wlist} are used.}

\item{wlist_covariates}{(Optional) List of matrices, which describes spatial dependencies for the covariates. If this is \code{NULL}, the matrices from \code{wlist} are used.}

\item{n_start}{Number of observations to be used for the burn-in period}

\item{control}{A list of parameters for controlling the fitting process. This list is passed to \code{\link{glmstarma.control}}.}
}
\value{
a named list with the following elements:
\itemize{
\item \code{observations} (numeric matrix): The simulated time series
\item \code{link_values} (numeric matrix): The underlying linear predictor resulting from the model and simulation
\item \code{model} (list): The model used for the simulation
\item \code{parameters} (list): The true parameters used for the simulation
}
}
\description{
Generates a simulated multivariate time series based on a GLM-like model (see \code{\link{glmstarma}} for details)
}
\examples{
set.seed(42)
n_obs <- 200L
W <- generateW("rectangle", 100, 2, 10)
model_orders <- list(intercept = "homogeneous", past_obs = 2, past_mean = 1)
parameter <- list(intercept = 0.5, past_obs = matrix(c(0.3, 0.2, 0.05), nrow = 3), 
                  past_mean = matrix(c(0.1, 0.05), nrow = 2),
                  covariates = c(0.75, 0.5))
covariates <- list(season = SpatialConstant(sin(2* pi / 12 * seq(n_obs))),
                   location = TimeConstant(rnorm(100, sd = 0.81)))
# Simulation using marginal poisson distribution
glmstarma.sim(n_obs, parameter, model_orders, W, covariates, family = vpoisson("log"))
# Simulation using negative binomial marginals
glmstarma.sim(n_obs, parameter, model_orders, W, covariates, 
              family = vnegative.binomial(dispersion = 3))
}
