% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ekw.R
\name{rekw}
\alias{rekw}
\title{Random Number Generation for the Exponentiated Kumaraswamy (EKw) Distribution}
\usage{
rekw(n, alpha = 1, beta = 1, lambda = 1)
}
\arguments{
\item{n}{Number of observations. If \code{length(n) > 1}, the length is
taken to be the number required. Must be a non-negative integer.}

\item{alpha}{Shape parameter \code{alpha} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{beta}{Shape parameter \code{beta} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{lambda}{Shape parameter \code{lambda} > 0 (exponent parameter).
Can be a scalar or a vector. Default: 1.0.}
}
\value{
A vector of length \code{n} containing random deviates from the EKw
distribution. The length of the result is determined by \code{n} and the
recycling rule applied to the parameters (\code{alpha}, \code{beta},
\code{lambda}). Returns \code{NaN} if parameters
are invalid (e.g., \code{alpha <= 0}, \code{beta <= 0}, \code{lambda <= 0}).
}
\description{
Generates random deviates from the Exponentiated Kumaraswamy (EKw)
distribution with parameters \code{alpha} (\eqn{\alpha}), \code{beta}
(\eqn{\beta}), and \code{lambda} (\eqn{\lambda}). This distribution is a
special case of the Generalized Kumaraswamy (GKw) distribution where
\eqn{\gamma = 1} and \eqn{\delta = 0}.
}
\details{
The generation method uses the inverse transform (quantile) method.
That is, if \eqn{U} is a random variable following a standard Uniform
distribution on (0, 1), then \eqn{X = Q(U)} follows the EKw distribution,
where \eqn{Q(u)} is the EKw quantile function (\code{\link{qekw}}):
\deqn{
Q(u) = \left\{ 1 - \left[ 1 - u^{1/\lambda} \right]^{1/\beta} \right\}^{1/\alpha}
}
This is computationally equivalent to the general GKw generation method
(\code{\link{rgkw}}) when specialized for \eqn{\gamma=1, \delta=0}, as the
required Beta(1, 1) random variate is equivalent to a standard Uniform(0, 1)
variate. The implementation generates \eqn{U} using \code{\link[stats]{runif}}
and applies the transformation above.
}
\examples{
\donttest{
set.seed(2027) # for reproducibility

# Generate 1000 random values from a specific EKw distribution
alpha_par <- 2.0
beta_par <- 3.0
lambda_par <- 1.5

x_sample_ekw <- rekw(1000, alpha = alpha_par, beta = beta_par, lambda = lambda_par)
summary(x_sample_ekw)

# Histogram of generated values compared to theoretical density
hist(x_sample_ekw,
  breaks = 30, freq = FALSE, # freq=FALSE for density
  main = "Histogram of EKw Sample", xlab = "x", ylim = c(0, 3.0)
)
curve(dekw(x, alpha = alpha_par, beta = beta_par, lambda = lambda_par),
  add = TRUE, col = "red", lwd = 2, n = 201
)
legend("topright", legend = "Theoretical PDF", col = "red", lwd = 2, bty = "n")

# Comparing empirical and theoretical quantiles (Q-Q plot)
prob_points <- seq(0.01, 0.99, by = 0.01)
theo_quantiles <- qekw(prob_points,
  alpha = alpha_par, beta = beta_par,
  lambda = lambda_par
)
emp_quantiles <- quantile(x_sample_ekw, prob_points, type = 7)

plot(theo_quantiles, emp_quantiles,
  pch = 16, cex = 0.8,
  main = "Q-Q Plot for EKw Distribution",
  xlab = "Theoretical Quantiles", ylab = "Empirical Quantiles (n=1000)"
)
abline(a = 0, b = 1, col = "blue", lty = 2)

# Compare summary stats with rgkw(..., gamma=1, delta=0, ...)
# Note: individual values will differ due to randomness
x_sample_gkw <- rgkw(1000,
  alpha = alpha_par, beta = beta_par, gamma = 1.0,
  delta = 0.0, lambda = lambda_par
)
print("Summary stats for rekw sample:")
print(summary(x_sample_ekw))
print("Summary stats for rgkw(gamma=1, delta=0) sample:")
print(summary(x_sample_gkw)) # Should be similar
}

}
\references{
Nadarajah, S., Cordeiro, G. M., & Ortega, E. M. (2012). The exponentiated
Kumaraswamy distribution. \emph{Journal of the Franklin Institute}, \emph{349}(3),

Cordeiro, G. M., & de Castro, M. (2011). A new family of generalized
distributions. \emph{Journal of Statistical Computation and Simulation},

Kumaraswamy, P. (1980). A generalized probability density function for
double-bounded random processes. \emph{Journal of Hydrology}, \emph{46}(1-2), 79-88.

Devroye, L. (1986). \emph{Non-Uniform Random Variate Generation}. Springer-Verlag.
(General methods for random variate generation).
}
\seealso{
\code{\link{rgkw}} (parent distribution random generation),
\code{\link{dekw}}, \code{\link{pekw}}, \code{\link{qekw}} (other EKw functions),
\code{\link[stats]{runif}}
}
\author{
Lopes, J. E.
}
\keyword{distribution}
\keyword{random}
