% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/density.R
\name{density_unbounded}
\alias{density_unbounded}
\title{Unbounded density estimator}
\usage{
density_unbounded(
  x,
  weights = NULL,
  n = 501,
  bandwidth = "dpi",
  adjust = 1,
  kernel = "gaussian",
  trim = TRUE,
  adapt = 1,
  na.rm = FALSE,
  ...,
  range_only = FALSE
)
}
\arguments{
\item{x}{<\link{numeric}> Sample to compute a density estimate for.}

\item{weights}{<\link{numeric} | \link{NULL}> Optional weights to apply to \code{x}.}

\item{n}{<scalar \link{numeric}> The number of grid points to evaluate the density estimator at.}

\item{bandwidth}{<scalar \link{numeric} | \link{function} | \link[=character]{string}>
Bandwidth of the density estimator. One of:
\itemize{
\item a numeric: the bandwidth, as the standard deviation of the kernel
\item a function: a function taking \code{x} (the sample) and returning the bandwidth
\item a string: the suffix of the name of a function starting with \code{"bandwidth_"} that
will be used to determine the bandwidth. See \link{bandwidth} for a list.
}}

\item{adjust}{<scalar \link{numeric}> Value to multiply the bandwidth of the density estimator by. Default \code{1}.}

\item{kernel}{<\link[=character]{string}> The smoothing kernel to be used. This must partially
match one of \code{"gaussian"}, \code{"rectangular"}, \code{"triangular"}, \code{"epanechnikov"},
\code{"biweight"}, \code{"cosine"}, or \code{"optcosine"}. See \code{\link[stats:density]{stats::density()}}.}

\item{trim}{<scalar \link{logical}> Should the density estimate be trimmed to the range of the data? Default \code{TRUE}.}

\item{adapt}{<positive \link{integer}> (\strong{very experimental}) The name and interpretation of this argument
are subject to change without notice. If \code{adapt > 1}, uses
an adaptive approach to calculate the density. First, uses the
adaptive bandwidth algorithm of Abramson (1982) to determine local (pointwise)
bandwidths, then groups these bandwidths into \code{adapt} groups, then calculates
and sums the densities from each group. You can set this to a very large number
(e.g. \code{Inf}) for a fully adaptive approach, but this will be very slow; typically
something around 100 yields nearly identical results.}

\item{na.rm}{<scalar \link{logical}> Should missing (\code{NA}) values in \code{x} be removed?}

\item{...}{Additional arguments (ignored).}

\item{range_only}{<scalar \link{logical}> If \code{TRUE}, the range of the output of this density estimator
is computed and is returned in the \verb{$x} element of the result, and \code{c(NA, NA)}
is returned in \verb{$y}. This gives a faster way to determine the range of the output
than \code{density_XXX(n = 2)}.}
}
\value{
An object of class \code{"density"}, mimicking the output format of
\code{\link[stats:density]{stats::density()}}, with the following components:
\itemize{
\item \code{x}: The grid of points at which the density was estimated.
\item \code{y}: The estimated density values.
\item \code{bw}: The bandwidth.
\item \code{n}: The sample size of the \code{x} input argument.
\item \code{call}: The call used to produce the result, as a quoted expression.
\item \code{data.name}: The deparsed name of the \code{x} input argument.
\item \code{has.na}: Always \code{FALSE} (for compatibility).
\item \code{cdf}: Values of the (possibly weighted) empirical cumulative distribution
function at \code{x}. See \code{\link[=weighted_ecdf]{weighted_ecdf()}}.
}

This allows existing methods for density objects, like \code{\link[=print]{print()}} and \code{\link[=plot]{plot()}}, to work if desired.
This output format (and in particular, the \code{x} and \code{y} components) is also
the format expected by the \code{density} argument of the \code{\link[=stat_slabinterval]{stat_slabinterval()}}
and the \code{\link[=smooth_density]{smooth_}} family of functions.
}
\description{
Unbounded density estimator using \code{\link[stats:density]{stats::density()}}.

Supports \link[=auto_partial]{automatic partial function application} with
\link[=waiver]{waived arguments}.
}
\examples{
library(distributional)
library(dplyr)
library(ggplot2)

# For compatibility with existing code, the return type of density_unbounded()
# is the same as stats::density(), ...
set.seed(123)
x = rbeta(5000, 1, 3)
d = density_unbounded(x)
d

# ... thus, while designed for use with the `density` argument of
# stat_slabinterval(), output from density_unbounded() can also be used with
# base::plot():
plot(d)

# here we'll use the same data as above, but pick either density_bounded()
# or density_unbounded() (which is equivalent to stats::density()). Notice
# how the bounded density (green) is biased near the boundary of the support,
# while the unbounded density is not.
data.frame(x) \%>\%
  ggplot() +
  stat_slab(
    aes(xdist = dist), data = data.frame(dist = dist_beta(1, 3)),
    alpha = 0.25
  ) +
  stat_slab(aes(x), density = "bounded", fill = NA, color = "#d95f02", alpha = 0.5) +
  stat_slab(aes(x), density = "unbounded", fill = NA, color = "#1b9e77", alpha = 0.5) +
  scale_thickness_shared() +
  theme_ggdist()
}
\seealso{
Other density estimators: 
\code{\link{density_bounded}()},
\code{\link{density_histogram}()}
}
\concept{density estimators}
