% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geostan_fit-methods.R
\name{residuals.geostan_fit}
\alias{residuals.geostan_fit}
\alias{fitted.geostan_fit}
\alias{spatial}
\alias{spatial.geostan_fit}
\title{Extract residuals, fitted values, or the spatial trend}
\usage{
\method{residuals}{geostan_fit}(object, summary = TRUE, rates = TRUE, detrend = TRUE, ...)

\method{fitted}{geostan_fit}(object, summary = TRUE, rates = TRUE, trend = TRUE, ...)

spatial(object, summary = TRUE, ...)

\method{spatial}{geostan_fit}(object, summary = TRUE, ...)
}
\arguments{
\item{object}{A fitted model object of class \code{geostan_fit}.}

\item{summary}{Logical; should the values be summarized by their mean, standard deviation, and quantiles (\code{probs = c(.025, .2, .5, .8, .975)}) for each observation? Otherwise, a matrix containing samples from the posterior distributions is returned.}

\item{rates}{For Poisson and Binomial models, should the fitted values be returned as rates, as opposed to raw counts? Defaults to \code{TRUE}; see the \code{Details} section for more information.}

\item{detrend}{For auto-normal models (CAR and SAR models with Gaussian likelihood only); if \code{detrend = FALSE}, the residuals will equal \code{y - Mu}, where \code{Mu} does not contain the implicit spatial trend. If \code{detrend = TRUE} (the default), the residuals will equal \code{Resid = y - Mu - Trend}. For CAR models and for SAR models of type SEM, the implicit spatial trend is \code{Trend = rho * C \%*\% (Y - Mu)} (see \link[geostan]{stan_car} or \link[geostan]{stan_sar}). For SAR models of type SLM, the spatial trend is \code{rho * C * y}.}

\item{...}{Not used}

\item{trend}{For auto-normal models (CAR and SAR models with Gaussian likelihood only); if \code{trend = FALSE}, the fitted values will not include the implicit spatial trend term. Rather, the fitted values will equal \code{Mu}: the intercept plus any other terms explicitly included in the model formula and the \code{re} argument ('random effects'). If \code{trend = TRUE} (the default), \code{fitted = Mu + Trend}. For CAR models and SAR models of type SEM, the spatial trend is \code{rho * C (y - Mu)}. For SAR models of type SLM, the spatial trend is \code{rho * C * y}.}
}
\value{
By default, these methods return a \code{data.frame}. The column named \code{mean} is what most users will be looking for. These contain the fitted values (for the \code{fitted} method), the residuals (fitted values minus observed values, for the \code{resid} method), or the spatial trend (for the \code{spatial} method). The \code{mean} column is the posterior mean of each value, and the column \code{sd} contains the posterior standard deviation for each value. The posterior distributions are also summarized by select quantiles.

If \code{summary = FALSE} then the method returns an S-by-N matrix of MCMC samples, where S is the number of MCMC samples and N is the number of observations in the data.

For spatial GLMs, the \code{spatial} method returns values on the scale of the linear predictor. Hence, for a Poisson with log link function, the spatial trend is on the log scale (in the same way that the intercept and coefficients are on the log scale).
}
\description{
Extract model residuals, fitted values, or spatial trend from a fitted \code{geostan_fit} model.
}
\details{
When \code{rates = FALSE} and the model is Poisson or Binomial, the fitted values returned by the \code{fitted} method are the expected value of the response variable. The \code{rates} argument is used to translate count outcomes to rates by dividing by the appropriate denominator. The behavior of the \code{rates} argument depends on the model specification. Consider a Poisson model of disease incidence, such as the following intercept-only case:

\if{html}{\out{<div class="sourceCode">}}\preformatted{fit <- stan_glm(y ~ offset(log(E)),
               data = data,
               family = poisson())
}\if{html}{\out{</div>}}

If the fitted values are extracted using \code{rates = FALSE}, then \code{fitted(fit)} will return the expectation of \eqn{y}. If \code{rates = TRUE} (the default), then \code{fitted(fit)} will return the expected value of the rate \eqn{\frac{y}{E}}.

If a binomial model is used instead of the Poisson, then using \code{rates = TRUE} with the \code{fitted} method will return the expectation of \eqn{\frac{y}{N}} where \eqn{y} is the number of 'successes' and \eqn{N} is the sum of the number of 'successes' and 'failures', as in:

\if{html}{\out{<div class="sourceCode">}}\preformatted{fit <- stan_glm(cbind(successes, failures) ~ 1,
               data = data,
               family = binomial())
}\if{html}{\out{</div>}}
}
\examples{
sar_list <- prep_sar_data2(row = 10, col = 10, quiet = TRUE)
W <- sar_list$W
x <- sim_sar(w = W, rho = 0.8)
y <- sim_sar(w = W, rho = 0.7, mu = 1 + 0.5 * x)
dat <- data.frame(x = x, y = y)

fit <- stan_sar(y ~ x, data = dat, sar = sar_list,
                    chains = 1, iter = 800)

# Residuals
r = resid(fit)
head(r)
moran_plot(r$mean, W)

# when residuals are not detrended, they exhibit spatial autocorrelation
r2 = resid(fit, detrend = FALSE)
mc(r$mean, W)
mc(r2$mean, W)

r_mat <- resid(fit, summary = FALSE)
r_mean <- apply(r_mat, 2, mean)
head(r_mean)
mean(r_mean)

# Fitted values
f <- fitted(fit)
head(f)

# Spatial trend
sp_trend  <- spatial(fit)
head(sp_trend)

# another way to obtain detrended residuals 
sp <- spatial(fit, summary = FALSE)
r <- resid(fit, detrend = FALSE, summary = FALSE)
# detrended residuals (matrix of samples):
r_det <- r - sp
# summarize (take posterior means):
r_det <- apply(r_det, 2, mean)
r <- apply(r, 2, mean)
# Moran coefficients:
mc(r_det, W)
mc(r, W)
}
