# set up quiet functions for testing reasons
purrr_compute <- purrr::quietly(compute.data_request)

# Warning generated by `galah_identify()` about empty query
test_that("`galah_identify()` returns an empty tibble when no args provided", {
  expect_warning({result <- galah_identify()})
  expect_equal(nrow(result), 0)
  expect_warning(galah_identify(), "No query passed")
})

test_that("`galah_identify()` runs a search when given a string", {
  result <- galah_identify("Litoria peronii")
  expect_equal(nrow(result), 1)
})

test_that("`galah_identify()` runs a search on multiple strings", {
  result <- galah_identify("amphibia", "reptilia", "aves", "mammalia")
  expect_equal(nrow(result), 4)
})

test_that("`galah_identify()` runs a search on multiple strings wrapped by c()", {
  search_terms <- c("amphibia", "reptilia", "aves", "mammalia")
  result <- galah_identify(search_terms)
  expect_equal(nrow(result), 4)
})

test_that("`identify()` pipes correctly", {
  result <- galah_call() |>
    identify("Litoria") |>
    filter(year == 2020)
  expect_false(is.null(result$identify))
  expect_false(is.null(result$filter))
})

test_that("`identify()` pipes correctly when taxa are partially returned", {
  galah_config(verbose = FALSE)
  result <- galah_call() |>
    identify("Litoria", "blarghy") |>
    filter(year == 2020)
  expect_false(is.null(result$identify))
  expect_false(is.null(result$filter))
  galah_config(verbose = TRUE)
})

test_that("`identify()` warns when taxa are partially returned", {
  skip_if_offline(); skip_on_ci()
  x <- galah_call() |>
    identify("Litoria", "blarghy") |>
    filter(year == 2020) |>
    count() |>
    purrr_compute() 
  x |>
    purrr::pluck("messages") |>
    stringr::str_detect("Matched 1 of 2 taxonomic search terms in selected atlas") |>
    any() |>
    expect_true()
})

test_that("`identify()` warns when identifiers are partially returned", {
  skip_if_offline(); skip_on_ci()
  galah_config(run_checks = TRUE)
  ids <- c("https://biodiversity.org.au/afd/taxa/0df99ece-1982-4605-a2b3-4fcb7660ee2b",
           "https://id.biodiversity.org.au/node/apni/2910467",
           "https://id.biodiversity.org.au/node/apni/291047") # wrong id
  x <- galah_call() |>
    identify(ids) |>
    filter(year == 2020) |>
    count() |>
    purrr_compute()
  x |>
    purrr::pluck("messages") |>
    stringr::str_detect("2 unmatched search terms:") |>
    any() |>
    expect_true()
  expect_no_warning(
    galah_identify(ids)
  )
})

test_that("`identify()` truncates unmatched list of taxa at 3 ", {
  skip_if_offline(); skip_on_ci()
  x <- galah_call() |>
    galah_identify("Litoria", "blarghy", "blorp", "florp", "skorp") |>
    galah_filter(year == 2020) |>
    count() |>
    purrr_compute()
  expected_messages <- c(
    "Matched 1 of 5 taxonomic search terms in selected atlas (Australia).\n",
    "4 unmatched search terms:\n",
    "\"blarghy\", \"blorp\", \"florp\" + 1 more\"\n",
    "\n")
  actual_messages <- purrr::pluck(x, "messages")
  # NOTE: due to differences in how dot points are represented, this test
  # fails unless we handle the strings carefully
  expect_equal(
    stringr::str_replace_all(actual_messages, "[:punct:]", "") |> trimws(),
    stringr::str_replace_all(expected_messages, "[:punct:]", "") |> trimws())
})

## NOTE: Not certain if this is a necessary test
# cli::test_that_cli("Partial taxonomic match message theming", {
#   testthat::local_edition(3)
#   testthat::expect_snapshot(local({
#     cli::cli_div(theme = list(span.bold = list("font-weight" = "bold"),
#                               span.yellow = list(color = "yellow")))
#     c(
#       cli::cli_text("Matched {.bold 2 of 3} taxonomic search terms in selected atlas (Australia)."),
#       "!" = cli::cli_text("{.yellow 1 unmatched search term:}"),
#       cli::cli_text(format_error_bullets(c("{.yellow \"https://id.biodiversity.org.au/node/apni/291047\"}")))
#     )
#   }))
# })

rm(purrr_compute)