% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nu_POP_estimator.R
\name{nu_POP_estimator}
\alias{nu_POP_estimator}
\title{Estimate the degrees of freedom of a heavy-tailed t distribution based on the POP estimator}
\usage{
nu_POP_estimator(
  Xc = NULL,
  N = NULL,
  T = NULL,
  Sigma = NULL,
  nu = NULL,
  r2 = NULL,
  method = c("POP", "POP-approx-1", "POP-approx-2", "POP-approx-3", "POP-approx-4",
    "POP-exact", "POP-sigma-corrected", "POP-sigma-corrected-true"),
  alpha = 1
)
}
\arguments{
\item{Xc}{Centered data matrix (with zero mean) containing the multivariate time series (each column is one time series).}

\item{N}{Number of variables (columns of data matrix) in the multivariate time series.}

\item{T}{Number of observations (rows of data matrix) in the multivariate time series.}

\item{Sigma}{Current estimate of the scatter matrix.}

\item{nu}{Current estimate of the degrees of freedom of the \eqn{t} distribution.}

\item{r2}{Vector containing the values of \code{diag( Xc \%*\% inv(scatter) \%*\% t(Xc) )}.}

\item{method}{String indicating the version of the POP estimator (default is just \code{"POP"} and should work well in all cases).
Other versions include: \code{"POP-approx-1"}, \code{"POP-approx-2"}, \code{"POP-approx-3"},
\code{"POP-approx-4"}, \code{"POP-exact"}, \code{"POP-sigma-corrected"}, \code{"POP-sigma-corrected-true"}.}

\item{alpha}{Value for the acceleration technique (cf. \code{\link{fit_mvt}()}).}
}
\value{
Estimated value of the degrees of freedom \code{nu} of a heavy-tailed \eqn{t} distribution.
}
\description{
This function estimates the degrees of freedom of a heavy-tailed \eqn{t} distribution based on
             the POP estimator from paper [Pascal-Ollila-Palomar, EUSIPCO2021, Alg. 1].
             Traditional nonparametric methods or likelihood methods provide erratic estimations of
             the degrees of freedom unless the number of observations is very large.
             The POP estimator provides a stable estimator based on random matrix theory.
             A number of different versions are provided, but the default POP method will most likely
             be the desired choice.
}
\examples{
library(mvtnorm)       # to generate heavy-tailed data
library(fitHeavyTail)

# parameters
N <- 5
T <- 100
nu_true <- 4           # degrees of freedom
mu_true <- rep(0, N)   # mean vector
Sigma_true <- diag(N)  # scatter matrix

# generate data
X <- rmvt(n = T, sigma = Sigma_true, delta = mu_true, df = nu_true)  # generate Student's t data
mu <- colMeans(X)
Xc <- X - matrix(mu, T, N, byrow = TRUE)    # center data

# usage #1
nu_POP_estimator(Xc = Xc, nu = 10, Sigma = Sigma_true)

# usage #2
r2 <- rowSums(Xc * (Xc \%*\% solve(Sigma_true)))
nu_POP_estimator(r2 = r2, nu = 10, N = N)

# usage #3
nu_POP_estimator(r2 = r2, nu = 10, N = N, method = "POP-approx-1")

}
\references{
Frédéric Pascal, Esa Ollila, and Daniel P. Palomar, "Improved estimation of the degree of freedom parameter of
multivariate t-distribution," in Proc. European Signal Processing Conference (EUSIPCO), Dublin, Ireland, Aug. 23-27, 2021.
<https://doi.org/10.23919/EUSIPCO54536.2021.9616162>
}
\author{
Frédéric Pascal, Esa Ollila, and Daniel P. Palomar
}
